/*
 * Copyright 2021, 2024 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "fsl_debug_console.h"
#include "fsl_sensor_common.h"
#include "fsl_p3t11xx.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*******************************************************************************
 * Variables
 ******************************************************************************/
/*******************************************************************************
 * Code
 ******************************************************************************/
status_t P3T11XX_WriteReg(p3t11xx_handle_t *handle, uint32_t regAddress, uint8_t *regData, size_t dataSize)
{
    status_t result = kStatus_Success;
    result          = handle->Sensor_WriteTransfer(handle->device_handle, regAddress, regData, dataSize);

    return (result == kStatus_Success) ? result : kStatus_Fail;
}

status_t P3T11XX_ReadReg(p3t11xx_handle_t *handle, uint32_t regAddress, uint8_t *regData, size_t dataSize)
{
    status_t result = kStatus_Success;
    result          = handle->Sensor_ReadTransfer(handle->device_handle, regAddress, regData, dataSize);

    return (result == kStatus_Success) ? result : kStatus_Fail;
}

status_t P3T1755_Init(p3t11xx_handle_t *handle, p3t11xx_config_t *config)
{
    uint8_t regVal[2];
    
    assert(handle != NULL);
    assert(config != NULL);
    assert(config->Sensor_WriteTransfer != NULL);
    assert(config->Sensor_ReadTransfer != NULL);

    handle->Sensor_WriteTransfer = config->Sensor_WriteTransfer;
    handle->Sensor_ReadTransfer  = config->Sensor_ReadTransfer;
    handle->device_handle        = config->device_handle;

    status_t result = kStatus_Success;

    if (config->isReset)
    {
        /* Set to interrupt mode instead of comparator mode, CONFIG is 8-bit. */
        regVal[0] = P3T1755_CONFIG_ENABLE_TM;
        result = P3T11XX_WriteReg(handle, CONFIG, &regVal[0], 1);
        if (result != kStatus_Success)
        {
            return result;
        }
        regVal[0] = P3T1755_TEMP_LOW >> 8;
        regVal[1] = P3T1755_TEMP_LOW & 0xFF;
        result = P3T11XX_WriteReg(handle, TEMP_LOW, &regVal[0], 2);
        if (result != kStatus_Success)
        {
            return result;
        }
        regVal[0] = P3T1755_TEMP_HIGH >> 8;
        regVal[1] = P3T1755_TEMP_HIGH & 0xFF;
        result = P3T11XX_WriteReg(handle, TEMP_HIGH, &regVal[0], 2);
        if (result != kStatus_Success)
        {
            return result;
        }
        SDK_DelayAtLeastUs(2000, SystemCoreClock);
    }
    return result;
}

status_t P3T1755_ReadSensorData(p3t11xx_handle_t *handle, uint32_t *sensorData)
{
    uint8_t temp_read[2];
    status_t result = kStatus_Success;

    result = P3T11XX_ReadReg(handle, TEMP, temp_read, 2);
    if (result == kStatus_Success)
    {
        *sensorData = ((temp_read[0] << 8) | temp_read[1]) >> 4;
    }
    return result;
}


status_t P3T1085_Init(p3t11xx_handle_t *handle, p3t11xx_config_t *config)
{
    uint8_t regVal[2];

    assert(handle != NULL);
    assert(config != NULL);
    assert(config->Sensor_WriteTransfer != NULL);
    assert(config->Sensor_ReadTransfer != NULL);

    handle->Sensor_WriteTransfer = config->Sensor_WriteTransfer;
    handle->Sensor_ReadTransfer  = config->Sensor_ReadTransfer;
    handle->device_handle        = config->device_handle;

    status_t result = kStatus_Success;

    if (config->isReset)
    {
        /* Set to interrupt mode instead of comparator mode, CONFIG is 16-bit. */
        regVal[0] = P3T1085_CONFIG_ENABLE_TM >> 8;
        regVal[1] = P3T1085_CONFIG_ENABLE_TM & 0xFF;
        result = P3T11XX_WriteReg(handle, CONFIG, &regVal[0], 2);
        if (result != kStatus_Success)
        {
            return result;
        }
        regVal[0] = P3T1085_TEMP_LOW >> 8;
        regVal[1] = P3T1085_TEMP_LOW & 0xFF;
        result = P3T11XX_WriteReg(handle, TEMP_LOW, &regVal[0], 2);
        if (result != kStatus_Success)
        {
            return result;
        }
        regVal[0] = P3T1085_TEMP_HIGH >> 8;
        regVal[1] = P3T1085_TEMP_HIGH & 0xFF;
        result = P3T11XX_WriteReg(handle, TEMP_HIGH, &regVal[0], 2);
        if (result != kStatus_Success)
        {
            return result;
        }
        SDK_DelayAtLeastUs(2000, SystemCoreClock);
    }
    return result;
}

status_t P3T1085_ReadSensorData(p3t11xx_handle_t *handle, uint32_t *sensorData)
{
    uint8_t temp_read[2];
    status_t result = kStatus_Success;

    result = P3T11XX_ReadReg(handle, TEMP, temp_read, 2);
    if (result == kStatus_Success)
    {
        *sensorData = ((temp_read[0] << 8) | temp_read[1]) >> 4;
    }
    return result;
}
