/*
 * Copyright 2016-2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/

#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "fsl_ctimer.h"
#include "fsl_lpadc.h"

#include "fsl_inputmux.h"
#include "fsl_power.h"

#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define CTIMER          CTIMER2         /* Timer 2 */
#define CTIMER_MAT3_OUT kCTIMER_Match_3 /* Match output 1 */
#define CTIMER_EMT3_OUT (1u << kCTIMER_Match_3)
#define CTIMER_CLK_FREQ CLOCK_GetCTimerClkFreq(2U)

#define DEMO_LPADC_BASE                  ADC0
#define DEMO_LPADC_USER_CMDID            1U                          /* CMD1 */
#define DEMO_LPADC_DO_OFFSET_CALIBRATION true
#define DEMO_LPADC_USE_HIGH_RESOLUTION   true

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
/* Match Configuration for Channel 0 */
static ctimer_match_config_t matchConfig0;

const uint32_t g_LpadcFullRange   = 65536U;
const uint32_t g_LpadcResultShift = 0U;

/*******************************************************************************
 * Code
 ******************************************************************************/



void CTIMER_SETUP(void)
{
	ctimer_config_t config;

	/* Use 96 MHz clock for Ctimer2 */
	CLOCK_SetClkDiv(kCLOCK_DivCtimer2Clk, 0u, false);
	CLOCK_SetClkDiv(kCLOCK_DivCtimer2Clk, 1u, true);
	CLOCK_AttachClk(kFRO_HF_to_CTIMER2);

	CTIMER_GetDefaultConfig(&config);

	CTIMER_Init(CTIMER, &config);

	/* Configuration 0 */
	matchConfig0.enableCounterReset = true;
	matchConfig0.enableCounterStop  = false;
	matchConfig0.matchValue         = CTIMER_CLK_FREQ / 2000;
	matchConfig0.outControl         = kCTIMER_Output_Toggle;
	matchConfig0.outPinInitState    = false;
	matchConfig0.enableInterrupt    = false;

	CTIMER_SetupMatch(CTIMER, CTIMER_MAT3_OUT, &matchConfig0);
	CTIMER_StartTimer(CTIMER);
}

void ADC_SETUP(void)
{
	lpadc_config_t mLpadcConfigStruct;
	lpadc_conv_trigger_config_t mLpadcTriggerConfigStruct;
	lpadc_conv_command_config_t mLpadcCommandConfigStruct;

	/*Attaching low frequency clock from PLL for calibration purposes: ADC0 Frequency at 5.7MHz*/
    CLOCK_SetClkDiv(kCLOCK_DivAdc0Clk, 0U, true);            /*!< Reset ADC0CLKDIV divider counter and halt it */
    CLOCK_SetClkDiv(kCLOCK_DivAdc0Clk, 8U, false);           /*!< Set ADC0CLKDIV divider to value 8 */
    CLOCK_AttachClk(kPLL0_to_ADC0);                 /*!< Switch ADC0 to PLL0 */

	/* Disable VREF power down */
	POWER_DisablePD(kPDRUNCFG_PD_VREF);

	LPADC_GetDefaultConfig(&mLpadcConfigStruct);
	mLpadcConfigStruct.enableAnalogPreliminary = true;
	mLpadcConfigStruct.referenceVoltageSource = kLPADC_ReferenceVoltageAlt3; /* VDDA */
	mLpadcConfigStruct.conversionAverageMode = kLPADC_ConversionAverage128;
	LPADC_Init(DEMO_LPADC_BASE, &mLpadcConfigStruct);

	LPADC_DoOffsetCalibration(DEMO_LPADC_BASE);
	/* Request gain calibration. */
	LPADC_DoAutoCalibration(DEMO_LPADC_BASE);

	/*Attaching Higher frequency Clock to ADC0 Frequency 48MHz*/
    CLOCK_SetClkDiv(kCLOCK_DivAdc0Clk, 0U, true);            /*!< Reset ADC0CLKDIV divider counter and halt it */
    CLOCK_SetClkDiv(kCLOCK_DivAdc0Clk, 2U, false);           /*!< Set ADC0CLKDIV divider to value 2 */
    CLOCK_AttachClk(kFRO_HF_to_ADC0);                 /*!< Switch ADC0 to PLL0 */

	/* Set conversion CMD configuration. */
	LPADC_GetDefaultConvCommandConfig(&mLpadcCommandConfigStruct);
	mLpadcCommandConfigStruct.channelNumber = 0U;
	mLpadcCommandConfigStruct.sampleChannelMode = kLPADC_SampleChannelSingleEndSideA;
	mLpadcCommandConfigStruct.conversionResolutionMode = kLPADC_ConversionResolutionHigh;

	LPADC_SetConvCommandConfig(DEMO_LPADC_BASE, DEMO_LPADC_USER_CMDID, &mLpadcCommandConfigStruct);

	/* Set trigger configuration. */
	LPADC_GetDefaultConvTriggerConfig(&mLpadcTriggerConfigStruct);
	mLpadcTriggerConfigStruct.targetCommandId       = 1U;
	mLpadcTriggerConfigStruct.enableHardwareTrigger = true;
	LPADC_SetConvTriggerConfig(DEMO_LPADC_BASE, 0U, &mLpadcTriggerConfigStruct); /* Configure the trigger0. */

}

void INPUTMUX_SETUP(void)
{
	INPUTMUX_Init(INPUTMUX);
	INPUTMUX_AttachSignal(INPUTMUX, 0U, kINPUTMUX_Ctimer2M3ToAdc0Trigger);
}

/*!
 * @brief Main function
 */
int main(void)
{
	lpadc_conv_result_t mLpadcResultConfigStruct;

	/* Init hardware*/
	/* attach 12 MHz clock to FLEXCOMM0 (debug console) */
	CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 0u, false);
	CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 1u, true);
	CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

	BOARD_InitPins();
	BOARD_BootClockPLL150M();
	BOARD_InitDebugConsole();

	INPUTMUX_SETUP();

	ADC_SETUP();

	CTIMER_SETUP();

	while(1)
	{
		while (!LPADC_GetConvResult(DEMO_LPADC_BASE, &mLpadcResultConfigStruct, 0U))
		{
		}
		PRINTF("ADC value: %d\r\n", ((mLpadcResultConfigStruct.convValue) >> g_LpadcResultShift));
	}


}
