/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "flexram_ecc.h"




#define FLEXRAM_ITCM_SIZE  0x20000
#define FLEXRAM_DTCM_SIZE  0x20000
#define FLEXRAM_OCRAM_SIZE 0x40000

void fill_flexram_itcm(void)
{
    volatile uint64_t * p = (uint64_t*)FLEXRAM_ITCM_START_ADDR;
    while((uint32_t)p < (FLEXRAM_ITCM_START_ADDR + FLEXRAM_ITCM_SIZE))
    {
        *p = 0;
        p++;
    }
}
void fill_flexram_dtcm(void)
{
    volatile uint64_t* p = (uint64_t*)FLEXRAM_DTCM_START_ADDR;
    while((uint32_t)p < (FLEXRAM_DTCM_START_ADDR + FLEXRAM_DTCM_SIZE))
    {
        *p = 0;
        p++;
    }
}
void fill_flexram_ocram(void)
{
    volatile uint64_t * p;

    p = (uint64_t*)FLEXRAM_OCRAM_START_ADDR;
    while((uint32_t)p < (FLEXRAM_OCRAM_START_ADDR + FLEXRAM_OCRAM_SIZE))
    {
        *p = 0;
        p++;
    }
}


#define PRINTFT DbgConsole_Printf_t
int DbgConsole_Printf_t(const char *fmt_s, ...);

void check_ecc(uint32_t addr)
{
    volatile uint32_t data32;

    PRINTFT("check_ecc.\r\n");
    PRINTFT("Press any key to continue \r\n\r\n"); 
    GETCHAR();
    data32 = *(volatile uint32_t*)addr;
    PRINTF("Check_ecc end\r\n");
    PRINTF("\r\n\r\n");
}

#define FLEXRAM_CONFIG_OCRAM 1
#define FLEXRAM_CONFIG_DTCM  2 
#define FLEXRAM_CONFIG_ITCM  3
void flexram_partition(void)
{
    IOMUXC_GPR->GPR17 = (FLEXRAM_CONFIG_OCRAM << 0)  |
                        (FLEXRAM_CONFIG_OCRAM << 2)  |
                        (FLEXRAM_CONFIG_OCRAM << 4)  |
                        (FLEXRAM_CONFIG_OCRAM << 6)  |
                        (FLEXRAM_CONFIG_OCRAM << 8)  |
                        (FLEXRAM_CONFIG_OCRAM << 10) |
                        (FLEXRAM_CONFIG_OCRAM << 12) |
                        (FLEXRAM_CONFIG_OCRAM << 14);

    IOMUXC_GPR->GPR18 = (FLEXRAM_CONFIG_ITCM << 0)  |
                        (FLEXRAM_CONFIG_ITCM << 2)  |
                        (FLEXRAM_CONFIG_ITCM << 4)  |
                        (FLEXRAM_CONFIG_ITCM << 6)  |
                        (FLEXRAM_CONFIG_DTCM << 8)  |
                        (FLEXRAM_CONFIG_DTCM << 10) |
                        (FLEXRAM_CONFIG_DTCM << 12) |
                        (FLEXRAM_CONFIG_DTCM << 14);

    IOMUXC_GPR->GPR16 |= IOMUXC_GPR_GPR16_FLEXRAM_BANK_CFG_SEL_MASK;

    PRINTF("IOMUXC_GPR->GPR16 = %x \r\n", IOMUXC_GPR->GPR16);
    PRINTF("IOMUXC_GPR->GPR17 = %x \r\n", IOMUXC_GPR->GPR17);
    PRINTF("IOMUXC_GPR->GPR18 = %x \r\n", IOMUXC_GPR->GPR18);
}


int main(void)
{
    uint32_t addr;
 
    /* Init board hardware. */ 
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
 
    PRINTF("\r\n\r\n");
    PRINTF("------------------------------------------------\r\n");
    PRINTF("FlexRAM ECC tests %s, %s\r\n", __DATE__, __TIME__);
    PRINTF("------------------------------------------------\r\n");
    PRINTF("\r\n\r\n");

    flexram_partition();
    flexram_ecc_init();

    fill_flexram_itcm();
    fill_flexram_dtcm();
    fill_flexram_ocram();

    PRINTF("\r\n\r\n");
    PRINTF("FLEXRAM->INT_STAT_EN >> 6 = %x\r\n",  FLEXRAM->INT_STAT_EN >> 6);
    PRINTF("FLEXRAM->INT_SIG_EN  >> 6 = %x\r\n",  FLEXRAM->INT_SIG_EN  >> 6);

    PRINTF("SCB->DTCMCR = %x\r\n",          SCB->DTCMCR);
    PRINTF("SCB->ITCMCR = %x\r\n",          SCB->ITCMCR);
    PRINTF("\r\n\r\n");

    PRINTF("------------------------------------------------\r\n");
    PRINTF("flexram - D0TCM test\r\n");
    PRINTF("------------------------------------------------\r\n");
    addr = 0x20000100;
    PRINTF("inject single error at addr = %x\r\n", addr);
    flexram_dtcm_error_injection_single(addr, 0);
    check_ecc(addr);

    PRINTF("inject multi error at addr = %x\r\n", addr);
    flexram_dtcm_error_injection_multi(addr, 0, 1);
    check_ecc(addr);

    PRINTF("------------------------------------------------\r\n");
    PRINTF("flexram - D1TCM test\r\n");
    PRINTF("------------------------------------------------\r\n");
    addr = 0x20000100;
    PRINTF("inject single error at addr = %x\r\n", addr);
    flexram_dtcm_error_injection_single(addr, 0);
    check_ecc(addr);

    PRINTF("inject multi error at addr = %x\r\n", addr);
    flexram_dtcm_error_injection_multi(addr, 0, 1);
    check_ecc(addr);

    PRINTF("------------------------------------------------\r\n");
    PRINTF("flexram - ITCM test\r\n");
    PRINTF("------------------------------------------------\r\n");
    addr = 0x100;
    PRINTF("inject single error at addr = %x\r\n", addr);
    flexram_itcm_error_injection_single(addr, 0);
    check_ecc(addr);

    PRINTF("inject multi error at addr = %x\r\n", addr);
    flexram_itcm_error_injection_multi(addr, 0, 1);
    check_ecc(addr);

    PRINTF("------------------------------------------------\r\n");
    PRINTF("flexram - OCRAM test\r\n");
    PRINTF("------------------------------------------------\r\n");
    addr = 0x20360100;
    PRINTF("inject single error at addr = %x\r\n", addr);
    flexram_ocram_error_injection_single(addr, 0);
    check_ecc(addr);

    PRINTF("inject multi error at addr = %x\r\n", addr);
    flexram_ocram_error_injection_multi(addr, 0, 1);
    check_ecc(addr);

    PRINTF("Example end. \r\n");
}

void BusFault_Handler(void)
{
    PRINTF("---------------BusFault_Handler \r\n");
    PRINTF("SCB->CFSR = %x \r\n", SCB->CFSR);
    PRINTF("SCB->BFAR = %x \r\n", SCB->BFAR);

    SCB->CFSR = 0xffffffff;
}







