
/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
 
#include "fsl_debug_console.h"
#include "flexram_ecc.h"

void flexram_ecc_init(void)
{
    SCB->DTCMCR |= SCB_DTCMCR_RMW_Msk | SCB_DTCMCR_RETEN_Msk;
    SCB->ITCMCR |= SCB_ITCMCR_RMW_Msk | SCB_ITCMCR_RETEN_Msk;

    NVIC_EnableIRQ(FLEXRAM_IRQn);
    NVIC_EnableIRQ(FLEXRAM_ECC_IRQn);
    
    FLEXRAM->FLEXRAM_CTRL = 0; 
    FLEXRAM->TCM_CTRL = 0;
 
    FLEXRAM->ITCM_ECC_ERROR_INJEC  = 0x0;
    FLEXRAM->D0TCM_ECC_ERROR_INJEC = 0x0;
    FLEXRAM->D1TCM_ECC_ERROR_INJEC = 0x0;
    FLEXRAM->INT_STATUS  = 0x0003FFFF;
    FLEXRAM->INT_SIG_EN  = 0x00000000;
    FLEXRAM->INT_STAT_EN = 0x00000000;

    // Status enable
    FLEXRAM->INT_STAT_EN =      FLEXRAM_INT_STAT_EN_ITCM_ERRM_INT_EN_MASK          | // itcm - multi bit
                                FLEXRAM_INT_STAT_EN_ITCM_ERRS_INT_EN_MASK          | // itcm - single bit
                                FLEXRAM_INT_STAT_EN_OCRAM_ERRM_INT_EN_MASK         | // ocram - multi bit
                                FLEXRAM_INT_STAT_EN_OCRAM_ERRS_INT_EN_MASK         | // ocram - single bit
                                FLEXRAM_INT_STAT_EN_D0TCM_ERRM_INT_EN_MASK         | // d0 tcm - multi bit
                                FLEXRAM_INT_STAT_EN_D0TCM_ERRS_INT_EN_MASK         | // d0 tcm - single bit
                                FLEXRAM_INT_STAT_EN_D1TCM_ERRM_INT_EN_MASK         | // d1 tcm - multi bit
                                FLEXRAM_INT_STAT_EN_D1TCM_ERRS_INT_EN_MASK         ; // d1 tcm - single bit

    // Interrupt enable.
    FLEXRAM->INT_SIG_EN  =      FLEXRAM_INT_SIG_EN_ITCM_ERRM_INT_SIG_EN_MASK        | // itcm - multi bit
                                FLEXRAM_INT_SIG_EN_ITCM_ERRS_INT_SIG_EN_MASK        | // itcm - single bit
                                FLEXRAM_INT_SIG_EN_OCRAM_ERRM_INT_SIG_EN_MASK       | // ocram - multi bit
                                FLEXRAM_INT_SIG_EN_OCRAM_ERRS_INT_SIG_EN_MASK       | // ocram - single bit
                                FLEXRAM_INT_SIG_EN_D0TCM_ERRM_INT_SIG_EN_MASK       | // d0 tcm - multi bit
                                FLEXRAM_INT_SIG_EN_D0TCM_ERRS_INT_SIG_EN_MASK       | // d0 tcm - single bit
                                FLEXRAM_INT_SIG_EN_D1TCM_ERRM_INT_SIG_EN_MASK       | // d1 tcm - multi bit
                                FLEXRAM_INT_SIG_EN_D1TCM_ERRS_INT_SIG_EN_MASK       ; // d1 tcm - single bit

    FLEXRAM->FLEXRAM_CTRL |= FLEXRAM_FLEXRAM_CTRL_TCM_ECC_EN_MASK |
                             FLEXRAM_FLEXRAM_CTRL_OCRAM_ECC_EN_MASK;

    __DSB();
}

#define ONE_SHOT_SINGLE (1<<16)
#define ONE_SHOT_MULTI  (1<<17)

static void d0tcm_error_injection_single(uint32_t address, uint32_t bit)
{
    address &= ~0x3;
    FLEXRAM->D0TCM_ECC_ERROR_INJEC = ONE_SHOT_SINGLE | (bit<<0);
     __DSB();
    *(uint32_t*)address = 0;
    __DSB();
    FLEXRAM->D0TCM_ECC_ERROR_INJEC = 0;
    __DSB();
}
static void d1tcm_error_injection_single(uint32_t address, uint32_t bit)
{
    address &= ~0x3;
    FLEXRAM->D1TCM_ECC_ERROR_INJEC = ONE_SHOT_SINGLE | (bit<<0);
     __DSB();
    *(uint32_t*)address = 0;
    __DSB();
    FLEXRAM->D1TCM_ECC_ERROR_INJEC = 0;
    __DSB();
}
static void d0tcm_error_injection_multi(uint32_t address, uint32_t bit0, uint32_t bit1)
{
    address &= ~0x3;
    FLEXRAM->D0TCM_ECC_ERROR_INJEC = ONE_SHOT_MULTI | (bit0<<0) | (bit1<<8);
     __DSB();
    *(uint32_t*)address = 0;
    __DSB();
    FLEXRAM->D0TCM_ECC_ERROR_INJEC = 0;
    __DSB();
}
static void d1tcm_error_injection_multi(uint32_t address, uint32_t bit0, uint32_t bit1)
{
    address &= ~0x3;
    FLEXRAM->D1TCM_ECC_ERROR_INJEC = ONE_SHOT_MULTI | (bit0<<0) | (bit1<<8);
     __DSB();
    *(uint32_t*)address = 0;
    __DSB();
    FLEXRAM->D1TCM_ECC_ERROR_INJEC = 0;
    __DSB();
}

void flexram_ocram_error_injection_single(uint32_t address, uint32_t bit)
{
    address &= ~0x7;

    FLEXRAM->OCRAM_ECC_ERROR_INJEC = ONE_SHOT_SINGLE | (bit<<0);
     __DSB();
    *(uint64_t*)address = 0;
    __DSB();
    FLEXRAM->OCRAM_ECC_ERROR_INJEC = 0;
    __DSB();
}
void flexram_ocram_error_injection_multi(uint32_t address, uint32_t bit0, uint32_t bit1)
{
    address &= ~0x7;

    FLEXRAM->OCRAM_ECC_ERROR_INJEC = ONE_SHOT_MULTI | (bit0<<0) | (bit1<<8);
     __DSB();
    *(uint64_t*)address = 0;
    __DSB();
    FLEXRAM->OCRAM_ECC_ERROR_INJEC = 0;
    __DSB();
}

void flexram_itcm_error_injection_single(uint32_t address, uint32_t bit)
{
    address &= ~0x7;

    FLEXRAM->ITCM_ECC_ERROR_INJEC = ONE_SHOT_SINGLE | (bit<<0);
     __DSB();
    *(uint64_t*)address = 0;
    __DSB();
    FLEXRAM->ITCM_ECC_ERROR_INJEC = 0;
    __DSB();
}
void flexram_itcm_error_injection_multi(uint32_t address, uint32_t bit0, uint32_t bit1)
{
    address &= ~0x7;

    FLEXRAM->ITCM_ECC_ERROR_INJEC = ONE_SHOT_MULTI | (bit0<<0) | (bit1<<8);
     __DSB();
    *(uint64_t*)address = 0;
    __DSB();
    FLEXRAM->ITCM_ECC_ERROR_INJEC = 0;
    __DSB();
}

void flexram_dtcm_error_injection_single(uint32_t address, uint32_t bit)
{
    if(address & 0x4)
    {
        d1tcm_error_injection_single(address, bit);
    }
    else
    {
        d0tcm_error_injection_single(address, bit);
    }
}

void flexram_dtcm_error_injection_multi(uint32_t address, uint32_t bit0, uint32_t bit1)
{
    if(address & 0x4)
    {
        d1tcm_error_injection_multi(address, bit0, bit1);
    }
    else
    {
        d0tcm_error_injection_multi(address, bit0, bit1);
    }
}


#define BASE_ADDR_FLEXRAM_DTCM  0x20000000
#define BASE_ADDR_FLEXRAM_ITCM  0x0
#define BASE_ADDR_FLEXRAM_OCRAM 0x20360000




/*
    Single bit error handler
*/
void FLEXRAM_IRQHandler(void)
{
    uint32_t status = FLEXRAM->INT_STATUS;

    FLEXRAM->INT_STATUS = 0x3FFFF;
    PRINTF("-->FLEXRAM_IRQHandler, status = %x\r\n", status);
    uint64_t * addr64;

    if (status & FLEXRAM_INT_STATUS_ITCM_ECC_ERRS_INT_MASK)
    {
        PRINTF("irq - flexram-itcm - single\r\n");
        PRINTF("FLEXRAM->ITCM_ECC_SINGLE_ERROR_INFO = %x \r\n",     FLEXRAM->ITCM_ECC_SINGLE_ERROR_INFO);
        PRINTF("FLEXRAM->ITCM_ECC_SINGLE_ERROR_ADDR = %x \r\n",     FLEXRAM->ITCM_ECC_SINGLE_ERROR_ADDR);
        PRINTF("Addr captured is %x\r\n", FLEXRAM_ITCM_START_ADDR + FLEXRAM->ITCM_ECC_SINGLE_ERROR_ADDR);
        PRINTF("FLEXRAM->ITCM_ECC_SINGLE_ERROR_DATA_LSB = %x \r\n", FLEXRAM->ITCM_ECC_SINGLE_ERROR_DATA_LSB);
        PRINTF("FLEXRAM->ITCM_ECC_SINGLE_ERROR_DATA_MSB = %x \r\n", FLEXRAM->ITCM_ECC_SINGLE_ERROR_DATA_MSB);
    }

    if (status & FLEXRAM_INT_STATUS_D0TCM_ECC_ERRS_INT_MASK)
    {
        PRINTF("irq - flexram-d0tcm - single\r\n");
        PRINTF("FLEXRAM->D0TCM_ECC_SINGLE_ERROR_INFO = %x \r\n", FLEXRAM->D0TCM_ECC_SINGLE_ERROR_INFO);
        PRINTF("FLEXRAM->D0TCM_ECC_SINGLE_ERROR_ADDR = %x \r\n", FLEXRAM->D0TCM_ECC_SINGLE_ERROR_ADDR);
        PRINTF("Addr captured is %x\r\n", FLEXRAM_DTCM_START_ADDR + FLEXRAM->D0TCM_ECC_SINGLE_ERROR_ADDR * 2);
        PRINTF("FLEXRAM->D0TCM_ECC_SINGLE_ERROR_DATA = %x \r\n", FLEXRAM->D0TCM_ECC_SINGLE_ERROR_DATA);
    }

    if (status & FLEXRAM_INT_STATUS_D1TCM_ECC_ERRS_INT_MASK)
    {
        PRINTF("irq - flexram-d1tcm - single\r\n");
        PRINTF("FLEXRAM->D1TCM_ECC_SINGLE_ERROR_INFO = %x \r\n", FLEXRAM->D1TCM_ECC_SINGLE_ERROR_INFO);
        PRINTF("FLEXRAM->D1TCM_ECC_SINGLE_ERROR_ADDR = %x \r\n", FLEXRAM->D1TCM_ECC_SINGLE_ERROR_ADDR);
        PRINTF("Addr captured is %x\r\n", FLEXRAM_DTCM_START_ADDR + FLEXRAM->D1TCM_ECC_SINGLE_ERROR_ADDR * 2);
        PRINTF("FLEXRAM->D1TCM_ECC_SINGLE_ERROR_DATA = %x \r\n", FLEXRAM->D1TCM_ECC_SINGLE_ERROR_DATA);
    }

    if (status & FLEXRAM_INT_STATUS_OCRAM_ECC_ERRS_INT_MASK)
    {
        PRINTF("irq - flexram-ocram - single\r\n");
        // Remove error injected for OCRAM
        addr64 = (uint64_t *)(BASE_ADDR_FLEXRAM_OCRAM + FLEXRAM->OCRAM_ECC_SINGLE_ERROR_ADDR);
        *addr64 = 0;
        __DSB();
        PRINTF("FLEXRAM->OCRAM_ECC_SINGLE_ERROR_INFO = %x \r\n",     FLEXRAM->OCRAM_ECC_SINGLE_ERROR_INFO);
        PRINTF("FLEXRAM->OCRAM_ECC_SINGLE_ERROR_ADDR = %x \r\n",     FLEXRAM->OCRAM_ECC_SINGLE_ERROR_ADDR);
        PRINTF("Addr captured is %x\r\n", BASE_ADDR_FLEXRAM_OCRAM + FLEXRAM->OCRAM_ECC_SINGLE_ERROR_ADDR);
        PRINTF("FLEXRAM->OCRAM_ECC_SINGLE_ERROR_DATA_LSB = %x \r\n", FLEXRAM->OCRAM_ECC_SINGLE_ERROR_DATA_LSB);
        PRINTF("FLEXRAM->OCRAM_ECC_SINGLE_ERROR_DATA_MSB = %x \r\n", FLEXRAM->OCRAM_ECC_SINGLE_ERROR_DATA_MSB);
    }
    PRINTF("-->Exit: FLEXRAM->INT_STATUS = %x\r\n", FLEXRAM->INT_STATUS);
}



/*
    Multi bit error handler
*/
void FLEXRAM_ECC_IRQHandler(void)
{
    uint32_t status;
    uint32_t * addr;
    uint64_t * addr64;

    status = FLEXRAM->INT_STATUS;
    FLEXRAM->INT_STATUS = 0x3FFFF;
    __DSB();

    PRINTF("-->FLEXRAM_ECC_IRQHandler, status = %x\r\n", status);
    PRINTF("FLEXRAM->INT_STATUS = %x\r\n", FLEXRAM->INT_STATUS);
    if (status & FLEXRAM_INT_STATUS_ITCM_ECC_ERRM_INT_MASK)
    {
        PRINTF("irq - itcm - multi\r\n");

        // Remove error injected
        addr64 = (uint64_t *)(BASE_ADDR_FLEXRAM_ITCM + FLEXRAM->ITCM_ECC_MULTI_ERROR_ADDR);
        PRINTF("Addr captured is %x\r\n", addr64);
        *addr64 = 0;
        __DSB();
        
        PRINTF("FLEXRAM->ITCM_ECC_MULTI_ERROR_INFO = %x \r\n",     FLEXRAM->ITCM_ECC_MULTI_ERROR_INFO);
        PRINTF("FLEXRAM->ITCM_ECC_MULTI_ERROR_ADDR = %x \r\n",     FLEXRAM->ITCM_ECC_MULTI_ERROR_ADDR);
        PRINTF("FLEXRAM->ITCM_ECC_MULTI_ERROR_DATA_LSB = %x \r\n", FLEXRAM->ITCM_ECC_MULTI_ERROR_DATA_LSB);
        PRINTF("FLEXRAM->ITCM_ECC_MULTI_ERROR_DATA_MSB = %x \r\n", FLEXRAM->ITCM_ECC_MULTI_ERROR_DATA_MSB);
    }   
    else if (status & FLEXRAM_INT_STATUS_D0TCM_ECC_ERRM_INT_MASK)
    {
        PRINTF("irq - d0tcm - multi\r\n");

        // Remove error injected
        addr = (uint32_t *)(BASE_ADDR_FLEXRAM_DTCM + FLEXRAM->D0TCM_ECC_MULTI_ERROR_ADDR * 2 + 0);
        PRINTF("Addr captured is %x\r\n", addr);        
        *addr = 0;
        __DSB();

        PRINTF("FLEXRAM->D0TCM_ECC_MULTI_ERROR_INFO = %x \r\n", FLEXRAM->D0TCM_ECC_MULTI_ERROR_INFO);
        PRINTF("FLEXRAM->D0TCM_ECC_MULTI_ERROR_ADDR = %x \r\n", FLEXRAM->D0TCM_ECC_MULTI_ERROR_ADDR);
        PRINTF("FLEXRAM->D0TCM_ECC_MULTI_ERROR_DATA = %x \r\n", FLEXRAM->D0TCM_ECC_MULTI_ERROR_DATA);
    }
    else if (status & FLEXRAM_INT_STATUS_D1TCM_ECC_ERRM_INT_MASK)
    {
        PRINTF("irq - d1tcm - multi\r\n");

        // Remove error injected
        addr = (uint32_t *)(BASE_ADDR_FLEXRAM_DTCM + FLEXRAM->D1TCM_ECC_MULTI_ERROR_ADDR * 2 + 4);
        PRINTF("Addr captured is %x\r\n", addr);        
        *addr = 0;
        __DSB();

        PRINTF("FLEXRAM->D1TCM_ECC_MULTI_ERROR_INFO = %x \r\n", FLEXRAM->D1TCM_ECC_MULTI_ERROR_INFO);
        PRINTF("FLEXRAM->D1TCM_ECC_MULTI_ERROR_ADDR = %x \r\n", FLEXRAM->D1TCM_ECC_MULTI_ERROR_ADDR);
        PRINTF("FLEXRAM->D1TCM_ECC_MULTI_ERROR_DATA = %x \r\n", FLEXRAM->D1TCM_ECC_MULTI_ERROR_DATA);
    }
    else if (status & FLEXRAM_INT_STATUS_OCRAM_ECC_ERRM_INT_MASK)
    {
        PRINTF("irq - flexram-ocram - multi\r\n");
        PRINTF("FLEXRAM->INT_STATUS = %x\r\n", FLEXRAM->INT_STATUS);

        // Remove error injected
        addr64 = (uint64_t *)(BASE_ADDR_FLEXRAM_OCRAM + FLEXRAM->OCRAM_ECC_MULTI_ERROR_ADDR);
        PRINTF("Addr captured is %x\r\n", addr64);
        *addr64 = 0;
        __DSB();

        PRINTF("FLEXRAM->OCRAM_ECC_MULTI_ERROR_INFO = %x \r\n",     FLEXRAM->OCRAM_ECC_MULTI_ERROR_INFO);
        PRINTF("FLEXRAM->OCRAM_ECC_MULTI_ERROR_ADDR = %x \r\n",     FLEXRAM->OCRAM_ECC_MULTI_ERROR_ADDR);
        PRINTF("FLEXRAM->OCRAM_ECC_MULTI_ERROR_DATA_LSB = %x \r\n", FLEXRAM->OCRAM_ECC_MULTI_ERROR_DATA_LSB);
        PRINTF("FLEXRAM->OCRAM_ECC_MULTI_ERROR_DATA_MSB = %x \r\n", FLEXRAM->OCRAM_ECC_MULTI_ERROR_DATA_MSB);
    }

    PRINTF("-->Exit: FLEXRAM->INT_STATUS = %x\r\n", FLEXRAM->INT_STATUS);
}












