/*
 * The Clear BSD License
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 *  that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_gpio.h"
#include "spiflash.h"
#include "flashiap.h"
#include "fsl_flashiap.h"

#include "pin_mux.h"
#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BLANK_VALUE_32B	(0xFFFFFFFF)

#define START_SBL_SEC (0)
#define START_SBL_ADDR (START_SBL_SEC*FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES)
#define START_APP_SEC (1)
#define START_APP_ADDR (START_APP_SEC*FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES)
#define END_APP_SEC	(FSL_FEATURE_SYSCON_FLASH_SIZE_BYTES/FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES - \
								    	START_APP_SEC) \
											

#define EXFLASH_HEAD_START_SEC (0)
#define	EXFLASH_HEAD_SIZE	(SECTOR_4KB_SIZE)//4K sector
#define EXFLASH_HEAD_START_ADDR	(FSL_FEATURE_SPIFI_START_ADDR + EXFLASH_HEAD_START_SEC*EXFLASH_HEAD_SIZE)
#define EXFLASH_IMAGE_START_ADDR (EXFLASH_HEAD_START_ADDR + EXFLASH_HEAD_SIZE)
/* definitions of the updated_flag in image header */
#define EXFLASH_APP_UPDATED_FLAG 0x5a5a5a5a	//flag shows image updated in external flash
#define INFLASH_APP_UPDATED_FLAG 0xa5a5a5a5//flag shows image updated in internal flash
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
typedef enum _firmware_update_status
{
		NOTHING_UPDATED,
		INFLASH_LASTED,
		EXFLASH_UPDATED,
		INFLASH_UPDATED
}fm_update_status_t;

typedef struct _image_header
{
		uint32_t image_size;//total size of firmware image
		uint32_t updated_flag;//flag of the image to be updated
		uint32_t version;//version of the firmware image
}image_hd_t;

__attribute__((aligned(4))) uint8_t page_buf[FSL_FEATURE_SYSCON_FLASH_PAGE_SIZE_BYTES] = {0};

//for app running
typedef  void (*pFunction)(void);
pFunction Jump_To_Application;
static uint32_t JumpAddress;
/*******************************************************************************
 * Code
 ******************************************************************************/
void run_to_app(uint32_t addr)
{
		JumpAddress = *(__IO uint32_t*) (addr + 4);

		/* Jump to user application */
		Jump_To_Application = (pFunction) JumpAddress;
		SCB->VTOR  = addr;
		/* Initialize user application's S4tack Pointer */
		__set_MSP(*(__IO uint32_t*) addr);
		__set_PSP(*(__IO uint32_t*) addr);
		Jump_To_Application();
}

/*!
 * @brief Main function
 */
int main(void)
{
		uint32_t status, result;
		uint32_t end_sect;
    uint32_t size;
    uint32_t spifi_addr, flash_addr;
    uint32_t offset;
		image_hd_t image_hd;
	
    /* Board pin, clock, debug console init */
    /* attach 12 MHz clock to FLEXCOMM2 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);
	
    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    BOARD_InitDebugConsole();
		
    /* Print a note to terminal. */
    PRINTF("\r\n\r\n         [WiFi tftp spifi OTA Demo - SBL]\r\n");
    
		spi_flash_init(kFRO_HF_to_SPIFI_CLK);
										
		spi_flash_setmode(SPI_FLASH_MODE_MEMORY);
		
		image_hd.updated_flag = *((volatile uint32_t *)(EXFLASH_HEAD_START_ADDR+4));
		if( *((volatile uint32_t *)(START_APP_ADDR)) == BLANK_VALUE_32B)//if APP area is blank
		{
				if(image_hd.updated_flag != EXFLASH_APP_UPDATED_FLAG)//if spifi flash is not updated
				{
						status = NOTHING_UPDATED;
				}					
				else //if app updated on spifi flash 
				{
						status = EXFLASH_UPDATED;
				}
		}
		else
		{
				if(image_hd.updated_flag == EXFLASH_APP_UPDATED_FLAG) //if app updated on spifi flash 
				{
						status = EXFLASH_UPDATED;		
				}
				else if(image_hd.updated_flag == INFLASH_APP_UPDATED_FLAG)
				{
						status = INFLASH_UPDATED;
				}
				else//if spifi flash is blank
				{
						status = INFLASH_LASTED;
				}	
		}
		
		switch(status)
		{
				case NOTHING_UPDATED:
						PRINTF("\r\nNo APP acquired! Please program it firstly!\r\n");
//						while(1);
				break;
				
				case EXFLASH_UPDATED://copy image from external spifi flash to internal flash
						PRINTF("New app found on spifi flash! Loading...\r\n");
						
						image_hd.image_size = *((volatile uint32_t *)(EXFLASH_HEAD_START_ADDR));
						end_sect = START_APP_SEC + image_hd.image_size/FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES;
						if((image_hd.image_size%FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES)==0)
						{
							end_sect -= 1;
						}
						if(end_sect > END_APP_SEC) end_sect = END_APP_SEC;
						
						result = prepare_programing(START_APP_SEC, end_sect);
						if(result != kStatus_FLASHIAP_Success)
						{
								PRINTF("Failed to prepare programing!\r\n");
								while(1);//halt
						}
						else
						{
								offset = 0;
								size = FSL_FEATURE_SYSCON_FLASH_PAGE_SIZE_BYTES;//access one page of size
								spifi_addr = EXFLASH_IMAGE_START_ADDR;
								flash_addr = START_APP_ADDR;

								while(offset < image_hd.image_size)
								{
										if((offset + FSL_FEATURE_SYSCON_FLASH_PAGE_SIZE_BYTES) > image_hd.image_size)
										{
												size = image_hd.image_size - offset;//get the last size which might be less than one page of size
										}
										memcpy(page_buf, (const void *)(spifi_addr+offset), size);//get data from spifi flash
										result = program_image_page(flash_addr+offset, page_buf);//program data to internal flash
										if(result != kStatus_FLASHIAP_Success)
										{
												PRINTF("Failed to program!\r\n");
												while(1);//halt
										}
										offset+=size;
								}	
								/* update flag */
								//update version info at spifi flash memory mode before changing to command mode
								image_hd.version = *((volatile uint32_t *)(EXFLASH_HEAD_START_ADDR+8));
								/* Set command on spifi flash */
								spi_flash_setmode(SPI_FLASH_MODE_COMMAND);
								image_hd.updated_flag = INFLASH_APP_UPDATED_FLAG;		
								memcpy(page_buf, (uint8_t *)&image_hd, sizeof(image_hd));
								spi_flash_erase_sector(EXFLASH_HEAD_START_SEC, SPI_FLASH_ERASE_SECTOR_4KB);
								spi_flash_program_page((EXFLASH_HEAD_START_SEC*EXFLASH_HEAD_SIZE)/PAGE_SIZE, page_buf);
						}
				case INFLASH_LASTED:													
				case INFLASH_UPDATED:
						if(status == INFLASH_LASTED)
							PRINTF("Old app running... as it is not updated correctly on spifi flash!\r\n");
						else
							PRINTF("running...\r\n");
						
						run_to_app(START_APP_ADDR);
				break;
					
				default:
//						while(1);//unexpected
				break;
		}
	
	while(1);
}
