/*
 * Copyright (c) 2013 - 2014, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

///////////////////////////////////////////////////////////////////////////////
//  Includes
///////////////////////////////////////////////////////////////////////////////

// SDK Included Files
#include "board.h"
#include "fsl_debug_console.h"
#include "qcom_api.h"
#include "wlan_qcom.h"
#include "wifi_driver_main.h"
#include "wlan_qcom_tftp.h"
#include "spiflash.h"
#include "flashiap.h"
#include "fsl_flashiap.h"

#include "pin_mux.h"
#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#if defined(LPCXPRESSO54608_EVB)
#define APP_VERSION	(0)//=0, LED1 shining in the version; =1, LED2 shining
#if APP_VERSION == 0
/* LED1 */
#define APP_BOARD_TEST_LED_PORT 3U
#define APP_BOARD_TEST_LED_PIN 14U
#else
/* LED2 */
#define APP_BOARD_TEST_LED_PORT 3U
#define APP_BOARD_TEST_LED_PIN 	3U
#endif
#elif defined(LPC5460X_IOT_KIT)
#define APP_VERSION	(0)//=0, red LED shining in the version; =1, green LED shining
#if APP_VERSION == 0
/* red LED */
#define APP_BOARD_TEST_LED_PORT 1U
#define APP_BOARD_TEST_LED_PIN 10U
#else
/* green LED */
#define APP_BOARD_TEST_LED_PORT 1U
#define APP_BOARD_TEST_LED_PIN 23U
#endif
#endif 
 
// 0 is the lowest priority and priority 15 is the highest priority
const int TASK_MAIN_PRIO = configMAX_PRIORITIES - 4;
const int TASK_MAIN_STACK_SIZE = 200;
const int TASK_OTA_PRIO = configMAX_PRIORITIES - 5;
const int TASK_OTA_STACK_SIZE = 800;
const int TASK_OTA_SETUP_PRIO = configMAX_PRIORITIES - 3;
const int TASK_OTA_SETUP_STACK_SIZE = 200;

portSTACK_TYPE *task_main_stack = NULL;
TaskHandle_t task_main_task_handler;
portSTACK_TYPE *task_ota_stack = NULL;
TaskHandle_t task_ota_task_handler;
portSTACK_TYPE *task_ota_setup_stack = NULL;
TaskHandle_t task_ota_setup_task_handler;

// Hardwired SSID, passphrase, auth and cipher algo of AP to connect to
#define AP_SSID ""
#define AP_PASSPHRASE ""

QCOM_SSID g_ssid = {.ssid = (AP_SSID)};
QCOM_PASSPHRASE g_passphrase = {.passphrase = (AP_PASSPHRASE)};

WLAN_AUTH_MODE g_auth = WLAN_AUTH_NONE;//WLAN_AUTH_WPA2_PSK;
WLAN_CRYPT_TYPE g_cipher = WLAN_CRYPT_NONE;//WLAN_CRYPT_AES_CRYPT;

#define APP_REV_NAME "revision.txt"
#define APP_IMG_NAME "app.bin"

//image header definitions on external flash. Note: keep same to SBL project!!!
#define EXFLASH_HEAD_START_SEC (0) //4K sector
#define EXFLASH_HEAD_START_PAGE (0)
#define	EXFLASH_HEAD_SIZE	(SECTOR_4KB_SIZE)//4K sector 
#define EXFLASH_HEAD_START_ADDR	(FSL_FEATURE_SPIFI_START_ADDR)
#define EXFLASH_IMAGE_START_ADDR (EXFLASH_HEAD_START_ADDR + EXFLASH_HEAD_SIZE)
#define EXFLASH_APP_UPDATED_FLAG 0x5a5a5a5a	//store in head of external flash
typedef struct _image_header
{
		uint32_t image_size;
		uint32_t updated_flag;
		uint32_t version;
}image_hd_t;
#define HD_VERSION_OFFSET	(sizeof(image_hd_t)-4)

extern int numIrqs;
extern int initTime;
extern QCOM_BSS_SCAN_INFO *scanResult;

enum STATE
{
    STATE_IDLE,
    STATE_CONNECTING,
    STATE_CONNECTED,
};
int state = STATE_IDLE;
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/
static bool apSelect(void)
{
		int ch, security, i;
		
		PRINTF("Select 'No' for one AP to connect :\r\n");
		PRINTF("Type '0','1'...(<10) or 'a','b'...for 10, 11... \r\n");
		PRINTF("(Or type 's' to scan again if AP not listed):");	
			
		//get ssid
		ch = GETCHAR();
		PRINTF("%c", ch);
		PRINTF("\r\n");
		if(ch=='s') 
		{	
				return false;
		}
		if(ch < 0x40)	i = ch - 0x30;
		else i = ch - 0x57;//'a','b'...for 10, 11...
		strcpy(g_ssid.ssid, (const char*)&scanResult[i].ssid);
		PRINTF("SSID = '%s'\r\n", g_ssid.ssid);
		security = scanResult[i].security_enabled;
	
		//input password
		PRINTF("Password:");
		i = 0;
		do
		{
				ch = GETCHAR();
				if(ch == 0x0D) 
				{	
						if(security)
						{
								g_auth = WLAN_AUTH_WPA2_PSK;
								g_cipher = WLAN_CRYPT_AES_CRYPT;
						}	
						break;
				}
				else
				{		
						if(security)
						{
								PRINTF("%c", ch);
								g_passphrase.passphrase[i++] = ch;
						}
				}
		}while(1);
		
		PRINTF("\r\n");	

		return true;
}

#define IP_DECI_LEN 3
extern uint8_t tftpIPAddr[4];
const uint8_t multiPlier[IP_DECI_LEN] = {100, 10, 1};
static void InputTFTPIP(void)
{
		int ch, i, j, k, p;
		uint8_t buf[IP_DECI_LEN+3];
	
	
		PRINTF("\r\nftp server IP address (format: xx.xx.xx.xx):");
		i = 0;
		j = 0;
		tftpIPAddr[j] = 0;
		do
		{
				ch = GETCHAR();
				if(ch!=0x0D&&ch!=0x1B)
				{
						PRINTF("%c", ch);
				}
				if((ch==0x2E)||(ch==0x0D))//if type '.' or CR
				{
						p = IP_DECI_LEN - i;
						for(k=0;k<i;k++)
						{
								tftpIPAddr[j] += buf[k]*multiPlier[p++];
						}		
						tftpIPAddr[++j] = 0;
						i = 0;
						if(ch == 0x0D) 
						{	
								PRINTF("\r\n");
								PRINTF("tftp server ip = %d.%d.%d.%d\r\n", tftpIPAddr[0],tftpIPAddr[1],
												tftpIPAddr[2],tftpIPAddr[3]);
								break;
						}
				}
				else if(ch==0x7f)//Backspace
				{
					  if(i)	
						{	
								buf[--i] = 0;
						}
						else
						{
								PRINTF("\r\nSorry! Please restart input by typing 'Esc'\r\n");
						}
				}
				else if(ch==0x1B)//Esc
				{
						i = 0;
						j = 0;
						tftpIPAddr[j] = 0;
						PRINTF("tftp server IP address (format: xx.xx.xx.xx):");
				}
				else
				{
						buf[i++] = ch-0x30;//valid char
				}	
		}while(1);
}

static void menuReqDHCP(void)
{
		getDhcp();
}
#define CONN_TIME_OUT_COUNT	50000000
static void menuConnectToAP(void)
{
		uint32_t i=0;
			
		apConnect(&g_ssid, &g_passphrase, g_auth, g_cipher);
		while(!isConnected())
		{
			i++;
			if(i==CONN_TIME_OUT_COUNT) 
			{	
					PRINTF("Time out!\r\n");
					break;
			}
		};
	  if(i<CONN_TIME_OUT_COUNT)
		{
				for(i=0;i<1000000;i++);//delay before request DHCP
				menuReqDHCP();
		}
}

static bool menuSelctAP(void)
{
		bool sel_ok=false;
		
	  do
		{
				PRINTF("---------Scaned AP List------------\r\n");
				apScan();	//scan avaible APs
				sel_ok = apSelect();//select one AP and input password if need
		}while(!sel_ok);
		return sel_ok;
}

static void menuDisconnect(void)
{
    apDisconnect();
}

static bool menuPingTFTPServer(void)
{
		PRINTF("tftp server ip = %d.%d.%d.%d\r\n", tftpIPAddr[0],tftpIPAddr[1],
												tftpIPAddr[2],tftpIPAddr[3]);
		PRINTF("Change the tftp IP address? [y or n]:");
		char ch = GETCHAR();
		PRINTF("%c\r\n", ch);
		if(ch == 'y')
		{//input the ip address
				InputTFTPIP();
		}
	
    int elapse = pingTFTP();
		if(elapse==-1) return false;
		
		PRINTF("Setup completed! Can exit with typing 'x'\r\n");
		return true;
}

static void menuViewIPConf(void)
{
		PRINTF("--------------------------------------\r\n");
		if(isConnected())
		{
				PRINTF("The AP connected!\r\n");
		}
		else
		{
				PRINTF("The AP disconnected!\r\n");
		}
		PRINTF("SSID = '%s'\r\n", g_ssid.ssid);
		PRINTF("password = '%s'\r\n", g_passphrase.passphrase);
		printIpConfig();
		PRINTF("--------------------------------------\r\n");
}

#define MAX_RETRY_NUM (5)
char *tftpRecvBuf = NULL;
int tftpRecvBufLen = 0;
int new_version;
#define BLANK_VALUE_32B	(0xFFFFFFFF)
//#define IMG_VERSION_MARK_OFFSET	(0)//(0x28)
//extern void *__Vectors;				
static bool check_tftp_new_version(void)
{
		bool rlt = false;
		uint8_t opcode;
		int old_version;
	
		PRINTF("\r\nChecking updated app on tftp server...\r\n");
		//send PRQ
		if(tftpPRQ(APP_REV_NAME)<0)
		{
				return rlt;
		}
		
		/* get a page of app file to check an updated version via tftp */
		if(tftpGet(10000, &tftpRecvBuf, &tftpRecvBufLen)<0)
		{
				return rlt;
		}
		opcode = tftpRecvBuf[1];
		
		rlt = 0;
		switch (opcode)
		{
			case OPCODE_DATA:
				//ack with block No. from data pack
				tftpACK(&tftpRecvBuf[2]);
			//check version mark
//				new_version = (tftpRecvBuf[4+IMG_VERSION_MARK_OFFSET]<<24)  
//														 + (tftpRecvBuf[5+IMG_VERSION_MARK_OFFSET]<<16)
//														 + (tftpRecvBuf[6+IMG_VERSION_MARK_OFFSET]<<8) 
//														 + tftpRecvBuf[7+IMG_VERSION_MARK_OFFSET];
//			
//				old_version = *((uint32_t *)(&__Vectors + IMG_VERSION_MARK_OFFSET));// version mark on internal flash
			
				spi_flash_setmode(SPI_FLASH_MODE_MEMORY);
				
				old_version = *((uint32_t *)(EXFLASH_HEAD_START_ADDR + HD_VERSION_OFFSET));//version mark on external spifi flash
				new_version = ((tftpRecvBuf[4]-0x30)<<24) + ((tftpRecvBuf[5]-0x30)<<16) 
										+ ((tftpRecvBuf[6]-0x30)<<8) + tftpRecvBuf[7]-0x30;
				if(old_version == BLANK_VALUE_32B)
				{
						old_version = 0;
				}
				if(new_version > old_version)
				{	
						rlt = true;			
				}				
				else rlt = false;
			
			break;
				
			case OPCODE_ERR:
				rlt = false;
			break;
			
			default:
				rlt = false;
			break;
		}
		tftpStop();		
		return rlt;
}
static void get_tftp_to_update(void)
{
		int rlt;
		bool is_completed;
		uint32_t sec_no;
		uint32_t page_no;
		uint8_t opcode;
		uint32_t i;
		static uint8_t blk_no[2] = {0};
		image_hd_t image_hd;	
		uint8_t buf[PAGE_SIZE] = {0xff};
		int retries;

		/* Set command on spifi flash */
		spi_flash_setmode(SPI_FLASH_MODE_COMMAND);
	
		//send PRQ
		PRINTF("\r\nSending request to tftp server...");
		rlt = tftpPRQ(APP_IMG_NAME);
	  if(rlt<0)
		{
				PRINTF("Failed!\r\n");
				return;
		}
		PRINTF("OK!\r\n");
		
		PRINTF("Getting app file to update spi flash...\r\n");
		/* keep getting and updating firmware to spifi flash via tftp */
		sec_no = 0;
		page_no = EXFLASH_HEAD_SIZE/PAGE_SIZE;
		image_hd.image_size = 0;
		is_completed = false;
		retries = 0;
		while(is_completed==false)
		{
				//get file content
				while((retries<MAX_RETRY_NUM)&&(tftpGet(5000, &tftpRecvBuf, &tftpRecvBufLen)<0))
				{
						retries++;
						PRINTF("Failed! retries=%d\r\n", retries);						
				}
				if(retries==MAX_RETRY_NUM) break;
				retries = 0;
				
				opcode = tftpRecvBuf[1];
				
				rlt = 0;
				switch (opcode)
				{
					case OPCODE_DATA:
						//ack with block No. from data pack
						while((retries<MAX_RETRY_NUM)&&(tftpACK(&tftpRecvBuf[2])<0))
						{
								retries++;
								PRINTF("ack retries=%d\r\n", retries);
						}
						if(retries==MAX_RETRY_NUM) 
						{
								rlt = 1;
								retries = 0;
								break;
						}
						if((tftpRecvBuf[2]==blk_no[0]) && (tftpRecvBuf[3]==blk_no[1]))
						{
								rlt = 1;//tftp server repeat tx
								break;
						}
						
						tftpRecvBufLen -= DATA_PACK_HD_LEN;
						image_hd.image_size += tftpRecvBufLen;
						if(tftpRecvBufLen < BLOCK_SIZE)//the package lenght is BLOCK_SIZE except the last one
						{
								is_completed = true;
						}
		
						blk_no[0] = tftpRecvBuf[2];
						blk_no[1] = tftpRecvBuf[3];
						
					break;
						
				  case OPCODE_ERR:
						rlt = -1;
					break;
					
					default:
						rlt = -1;
					break;
				}
				if(rlt<0) 
				{	
						PRINTF("Failed! error code = 0x%x%x\r\n", tftpRecvBuf[2],tftpRecvBuf[3]);
						return;
				}
				else if(rlt==0)//update spifi flash with image from tftp
				{		
						uint32_t total_size = image_hd.image_size+EXFLASH_HEAD_SIZE;
						uint32_t sec_num = total_size/SECTOR_SIZE;
						if(total_size%SECTOR_SIZE)
						{
								sec_num++;
						}
						if(sec_no<sec_num)
						{
								spi_flash_erase_sector(sec_no, SPI_FLASH_ERASE_SECTOR_64KB);
								sec_no += 1;														
						}					
				
						//all the page num to be programmed is right equal to 2 except for the last package			
						uint32_t page_num = tftpRecvBufLen/PAGE_SIZE;
						if(tftpRecvBufLen%PAGE_SIZE)
						{
								page_num++;
						}
						
						i=0;
						while(i<page_num)
						{
								spi_flash_program_page(page_no+i, (uint8_t *)&tftpRecvBuf[4+i*PAGE_SIZE]);
								i++;							
						}
						page_no += page_num;
				}
				
		}//while(is_completed==false)
		if(is_completed == true)
		{
				//update flag
				image_hd.updated_flag = EXFLASH_APP_UPDATED_FLAG;	
				image_hd.version = new_version;
				memcpy(buf, (uint8_t *)&image_hd, sizeof(image_hd));
		//		spi_flash_erase_sector(EXFLASH_HEAD_START_SEC, SPI_FLASH_ERASE_SECTOR_4KB);
				spi_flash_program_page(EXFLASH_HEAD_START_PAGE, buf);
				
				PRINTF("\r\nCompleted! file size=%d bytes\r\n", image_hd.image_size);
		}		

		tftpStop();
		return;
}

static void menuPrint(void);
//void menuClrUpdateInfo(void)
//{
//	spi_flash_setmode(SPI_FLASH_MODE_COMMAND);
//	spi_flash_erase_sector(EXFLASH_HEAD_START_SEC, SPI_FLASH_ERASE_SECTOR_4KB);
//}
typedef struct
{
    char key;
    void (*func)(void);
    const char *text;
} menu_item_t;
menu_item_t menuItems[] = {
		{'s', (void (*)(void))menuSelctAP, "scan & select AP"},
    {'c', menuConnectToAP, "connect AP"},
		{'p', (void (*)(void))menuPingTFTPServer, "Ping tftp server IP"},
		{'r', menuReqDHCP, "request DHCP address"},
		{'v', menuViewIPConf, "view IP configuration"},
    {'D', menuDisconnect, "disconnect AP"},
    {'m', menuPrint, "Print this menu"},
//		{'t', menuClrUpdateInfo, "Clear update info for test"},
		{'x', NULL, "Exit setup\r\n"},
		{0, NULL, NULL},
};

static void menuAction(int ch)
{
    for (int i = 0; menuItems[i].func != NULL; i++)
    {
        if (menuItems[i].key == ch)
        {
            PRINTF("Key '%c': %s\r\n", ch, menuItems[i].text);
            menuItems[i].func();
            return;
        }
    }
    PRINTF("ERROR: No action bound to '%c'\r\n", ch);
}

static void menuPrint(void)
{
    PRINTF("============ WiFi Network Setup Menu ============\r\n");

    for (int i = 0; menuItems[i].text != NULL; i++)
    {
        if (menuItems[i].key)
            PRINTF("  %c  %s\r\n", menuItems[i].key, menuItems[i].text);
        else
            PRINTF("  %d  %s\r\n", i, menuItems[i].text);
    }

}

static int pollChar()
{
    //    if (!UART_HAL_GetStatusFlag(BOARD_DEBUG_UART_BASEADDR, kUartRxDataRegFull))
    //        return -1;
    //
    //    uint8_t ch;
    //    UART_HAL_Getchar(BOARD_DEBUG_UART_BASEADDR, &ch);
    //    // Flush the UART to avoid the "OpenSDA UART infinite RX FIFO" bug:
    //    // If user presses keys very quickly then after a while, the UART enters
    //    // a mode where the same sequence of characters are repeatedly returned
    //    // by UART_HAL_Getchar(). Even a hard reset does not fix it.
    //    // Only way to recover is by unplugging OpenSDA USB.
    //    UART_HAL_FlushRxFifo(BOARD_DEBUG_UART_BASEADDR);

    int tmp;
    // TODO: verify the message above. Which board has this issue ?
    tmp = GETCHAR();
    return tmp;
}

// ============================================================================
// Main
// ============================================================================

static void print_version(void)
{
    ATH_VERSION_STR verstr;

    int res = qcom_get_versionstr(&verstr);
    if (A_OK == res)
    {
        PRINTF("Host version:      %s\r\n", verstr.host_ver);
        PRINTF("Target version:    %s\r\n", verstr.target_ver);
        PRINTF("Firmware version:  %s\r\n", verstr.wlan_ver);
        PRINTF("Interface version: %s\r\n", verstr.abi_ver);
    }
    else
    {
        PRINTF("ERROR: Failed to get QCA400X version information\r\n");
    }
}
bool check_network_set(void)
{
		if(!isConnected()) return false;//if connected
		
		if(!isIPAssigned()) return false;//if dhcp assigend
	
		if(!isTFTPPinged()) return false;//if tftp server pinged
	
		return true;
}
void task_setup_wifi(void *param)
{
		int result;
	
		PRINTF("\r\n-----set up WiFi network------\r\n");
		
		/* Initialize WIFI shield */
		result = WIFISHIELD_Init();
		assert(!(A_OK != result));
		if (A_OK != result) while(1);

		/* Initialize the WIFI driver (thus starting the driver task) */
		result = wlan_driver_start();
		assert(!(A_OK != result));
		if (A_OK != result) while(1);

		print_version();

		UBaseType_t numTasks = uxTaskGetNumberOfTasks();
		PRINTF("number of FreeRTOS tasks = %d\r\n", numTasks);
		PRINTF("Entering main loop\r\n");
	
		menuPrint();
	
		while (1)
		{							
				int ch = pollChar();
				if(ch == 'x') break;
				else if (ch != -1)
				{
						menuAction(ch);
				}
		}
		PRINTF("Exited network setup!\r\n");
		vTaskDelete(task_ota_setup_task_handler);
}
#define TASK_MIAN_DELAY_NUM (15000000)
void task_main(void *param)
{
	PRINTF("\r\nmain task: The LED is shining...\r\n\r\n");

    while (1)
    {
        GPIO_PortToggle(GPIO, APP_BOARD_TEST_LED_PORT, 1u << APP_BOARD_TEST_LED_PIN);
				for(uint32_t i=0;i<TASK_MIAN_DELAY_NUM;i++);//delay
				GPIO_PortToggle(GPIO, APP_BOARD_TEST_LED_PORT, 1u << APP_BOARD_TEST_LED_PIN);
				for(uint32_t i=0;i<TASK_MIAN_DELAY_NUM;i++);//delay
				GPIO_PortToggle(GPIO, APP_BOARD_TEST_LED_PORT, 1u << APP_BOARD_TEST_LED_PIN);
				for(uint32_t i=0;i<TASK_MIAN_DELAY_NUM;i++);//delay
				GPIO_PortToggle(GPIO, APP_BOARD_TEST_LED_PORT, 1u << APP_BOARD_TEST_LED_PIN);
				for(uint32_t i=0;i<TASK_MIAN_DELAY_NUM;i++);//delay
			
				vTaskDelay(MSEC_TO_TICK(80));
    }
}

void task_ota(void *param)
{
    while (1)
		{							
				if(check_network_set() == true)
				{
						if(check_tftp_new_version() == true)//if new version found on tftp server
						{
								PRINTF("\r\nNew app found!\r\n");
								get_tftp_to_update();
						}
				}
				vTaskDelay(MSEC_TO_TICK(50));
		}
}

int main(void)
{
    BaseType_t result = 0;
    (void)result;
	
		
		/* Define the init structure for the output LED pin*/
    gpio_pin_config_t led_config = {
        kGPIO_DigitalOutput, 1,
    };


    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    CLOCK_EnableClock(kCLOCK_InputMux);

    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    BOARD_InitDebugConsole();
	
		PRINTF("\r\n           [WiFi tftp spifi OTA Demo - APP]\r\n");
	
		 /* Init output LED GPIO. */
    GPIO_PortInit(GPIO, APP_BOARD_TEST_LED_PORT);
    GPIO_PinInit(GPIO, APP_BOARD_TEST_LED_PORT, APP_BOARD_TEST_LED_PIN, &led_config);
		
		/* initialize spi flash for OTA */
		spi_flash_init(kFRO_HF_to_SPIFI_CLK);		
		
    result =
        xTaskCreate(task_main, "main", TASK_MAIN_STACK_SIZE, task_main_stack, TASK_MAIN_PRIO, &task_main_task_handler);
    assert(pdPASS == result);
	
		result =
        xTaskCreate(task_ota, "ota", TASK_OTA_STACK_SIZE, task_ota_stack, TASK_OTA_PRIO, &task_ota_task_handler);
    assert(pdPASS == result);
		
		result =
        xTaskCreate(task_setup_wifi, "ota_setup", TASK_OTA_SETUP_STACK_SIZE, task_ota_setup_stack, TASK_OTA_SETUP_PRIO, &task_ota_setup_task_handler);
    assert(pdPASS == result);
	
    vTaskStartScheduler();
    for (;;)
        ;
}
