/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <stdlib.h>

#include "fsl_common.h"
#include "board.h"

#include "cmd_shell.h"
#include "i2c_comm.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define CMD_I2C_SLAVE_ADDRESS  0x45
#define CMD_I2C_MASTER_INSTANCE  I2C0

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void cmd_i2c_master_init(void);

/* Cmd Functions. */
int32_t cmd_show_help(int32_t argc, char *argv[]);
int32_t cmd_i2c_master_spi_write(int32_t argc, char *argv[]);
int32_t cmd_i2c_master_spi_read(int32_t argc, char *argv[]);
int32_t cmd_i2c_master_clear_buff(int32_t argc, char *argv[]);

/*******************************************************************************
 * Variables
 ******************************************************************************/
extern const CMD_HandlerCallback_T CmdHandlerCallbackStruct;

/* Cmd Table. */
CMD_TableItem_T CmdI2CMasterTestCmdsTable[] =
{
    {"help",   1, cmd_show_help},
    {"spi_write",  8, cmd_i2c_master_spi_write},
    {"spi_read",  1, cmd_i2c_master_spi_read},
    {"clear_buff", 1, cmd_i2c_master_clear_buff},
    {NULL}
};

#define CMD_I2C_MASTER_XFER_BUFF_LEN  (200u)
uint8_t cmd_i2c_master_xfer_buff[CMD_I2C_MASTER_XFER_BUFF_LEN]; /* the func code is at the first byte. */

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    board_init();
    printf("app i2c master.\r\n");

    cmd_i2c_master_init();

    CMD_InstallHandler(&CmdHandlerCallbackStruct);

    while (1)
    {
        CMD_LoopShell(CmdI2CMasterTestCmdsTable);
    }
}

void cmd_i2c_master_init(void)
{
    /* pins are already setup in BOARD_InitPins() by config tool. */

    /* i2c controller. */
    i2c_init_master(CMD_I2C_MASTER_INSTANCE, eI2C_Speed_400k);
}

/*******************************************************************************
 * Cmd Functions.
 ******************************************************************************/
int32_t cmd_show_help(int32_t argc, char *argv[])
{
    uint32_t index = 0u;
    printf("# cmd_show_help()\r\n\r\n");
    while ( CmdI2CMasterTestCmdsTable[index].CmdName != NULL )
    {
        printf(" - %s\r\n", CmdI2CMasterTestCmdsTable[index].CmdName);
        index++;
    }
    printf("\r\n");
    return 0;
}

int32_t cmd_i2c_master_spi_write(int32_t argc, char *argv[])
{
    for (uint32_t i = 0u; i < argc-1; i++)
    {
        cmd_i2c_master_xfer_buff[i] = atoi(argv[i+1]);
        printf("0x%-2X, ", cmd_i2c_master_xfer_buff[i]);
    }
    printf("\r\n");

    i2c_master_write(CMD_I2C_MASTER_INSTANCE, CMD_I2C_SLAVE_ADDRESS, cmd_i2c_master_xfer_buff, argc-1u);
    printf("i2c_master_write() done.\r\n");

    return 0;
}

int32_t cmd_i2c_master_spi_read(int32_t argc, char *argv[])
{
    i2c_master_read(CMD_I2C_MASTER_INSTANCE, CMD_I2C_SLAVE_ADDRESS, cmd_i2c_master_xfer_buff, CMD_I2C_MASTER_XFER_BUFF_LEN);
    printf("i2c_master_read() done.\r\n");

    for (uint32_t i = 0u; i < CMD_I2C_MASTER_XFER_BUFF_LEN; i++)
    {
        if ( (i % 10u) == 0u)
        {
            printf("\r\n");
        }
        printf("0x%-2X, ", cmd_i2c_master_xfer_buff[i]);
    }
    printf("\r\n");

    return 0;
}

int32_t cmd_i2c_master_clear_buff(int32_t argc, char *argv[])
{
    for (uint32_t i = 0u; i < CMD_I2C_MASTER_XFER_BUFF_LEN; i++)
    {
        cmd_i2c_master_xfer_buff[i] = 0u;
    }
    printf("cmd_i2c_mastert_clear_buff() done. \r\n");

    return 0;
}

/* EOF. */

