/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_common.h"
#include "fsl_power.h"
#include "fsl_inputmux.h"
#include "fsl_pint.h"
#include "fsl_usart.h"
#include "pmic_support.h"

#include "pin_mux.h"
#include "clock_config.h"
#include "fsl_pca9420.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*
 * The option is used for measurement of the deep sleep wake-up latency. When the application calls
 * POWER_EnterDeepSleep(), the power library will control the PLL power depending on the
 * exclude_from_pd parameter and the PLL running status. If the PLL is not in the exclude_from_pd list
 * and is running, it will be powered off before WFI and restored after WFI by the power library. Otherwise,
 * the PLL control will not be changed in POWER_EnterDeepSleep().
 * As the PLL initialization costs time to stabilize the clock, user will see the duration of
 * POWER_EnterDeepSleep() is longer than expectation.
 * To get rid of the PLL clock initialization time from deep sleep wake-up latency measurement, we can set
 * POWER_DOWN_PLL_BEFORE_DEEP_SLEEP to 1, then the demo itself will disable the PLL before
 * calling POWER_EnterDeepSleep(), and restore PLL after that. Thus we get the real duration of system
 * deep sleep wake-up time.
 */
#define POWER_DOWN_PLL_BEFORE_DEEP_SLEEP \
    0 /* By default, we keep the application as simple as possible to make good OOBE. */

#define SLEEP_AT_48M_IRC_DIV4 \
    0
#define WAKE_UP_WITH_PMIC_IRQ_N \
    0
#define APP_USART_RX_ERROR kUSART_RxError
#define APP_USER_WAKEUP_KEY_NAME "SW1"
#define APP_USER_WAKEUP_KEY_GPIO BOARD_SW1_GPIO
#define APP_USER_WAKEUP_KEY_PORT BOARD_SW1_GPIO_PORT
#define APP_USER_WAKEUP_KEY_PIN BOARD_SW1_GPIO_PIN
#define APP_USER_WAKEUP_KEY_INPUTMUX_SEL kINPUTMUX_GpioPort1Pin1ToPintsel
/*!< Power down all unnecessary blocks and enable RBB during deep sleep. */
#if 1
#define APP_DEEPSLEEP_RUNCFG0 (SYSCTL0_PDSLEEPCFG0_RBB_PD_MASK)
#else
#define APP_DEEPSLEEP_RUNCFG0 (SYSCTL0_PDSLEEPCFG0_RBB_PD_MASK | SYSCTL0_PDRUNCFG0_SYSXTAL_PD_MASK)
#endif
#define APP_DEEPSLEEP_RAM_APD 0xFFFFF8U
#define APP_DEEPSLEEP_RAM_PPD 0x0U
#define APP_EXCLUDE_FROM_DEEPSLEEP                                                                                \
    (((const uint32_t[]){APP_DEEPSLEEP_RUNCFG0,                                                                   \
                         (SYSCTL0_PDSLEEPCFG1_FLEXSPI_SRAM_APD_MASK | SYSCTL0_PDSLEEPCFG1_FLEXSPI_SRAM_PPD_MASK), \
                         APP_DEEPSLEEP_RAM_APD, APP_DEEPSLEEP_RAM_PPD}))

#define APP_EXCLUDE_FROM_DEEP_POWERDOWN (((const uint32_t[]){0, 0, 0, 0}))
#define APP_EXCLUDE_FROM_FULL_DEEP_POWERDOWN (((const uint32_t[]){0, 0, 0, 0}))
const char *gWakeupInfoStr[] = {"Sleep [Press the user key to wakeup]", "Deep Sleep [Press the user key to wakeup]",
#if (defined(FSL_FEATURE_SYSCON_HAS_POWERDOWN_MODE) && FSL_FEATURE_SYSCON_HAS_POWERDOWN_MODE)
                                "Powerdown [Reset to wakeup]", "Deep Powerdown [Reset to wakeup]"};
#else
                                "Deep Powerdown [Reset to wakeup]", "Full Deep Powerdown [Reset to wakeup]"};
#endif
uint32_t gCurrentPowerMode;
volatile bool pintFlag = false;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void BOARD_ConfigPMICModes(pca9420_modecfg_t *cfg, uint32_t num);
#if POWER_DOWN_PLL_BEFORE_DEEP_SLEEP
void BOARD_DisablePll();
void BOARD_RestorePll();
#endif
static uint32_t APP_GetUserSelection(void);
static void APP_InitWakeupPin(void);
static void pint_intr_callback(pint_pin_int_t pintr, uint32_t pmatch_status);

/*******************************************************************************
 * Code
 ******************************************************************************/
/*PLL status*/
extern const clock_sys_pll_config_t g_sysPllConfig_BOARD_BootClockRUN;
extern const clock_audio_pll_config_t g_audioPllConfig_BOARD_BootClockRUN;
AT_QUICKACCESS_SECTION_CODE(void BOARD_SetFlexspiClock(uint32_t src, uint32_t divider));


void PMC_PMIC_DriverIRQHandler(void)
{
    uint32_t statusMask;

    statusMask = POWER_GetEventFlags();
    POWER_ClearEventFlags(statusMask);
}

void BOARD_ConfigPMICModes(pca9420_modecfg_t *cfg, uint32_t num)
{
    assert(cfg);

    /* Configuration PMIC mode to align with power lib like below:
     *  0b00    run mode, no special.
     *  0b01    deep sleep mode, vddcore 0.7V.
     *  0b10    deep powerdown mode, vddcore off.
     *  0b11    full deep powerdown mode vdd1v8 and vddcore off. */

    /* Mode 1: VDDCORE 0.7V. */
    cfg[1].sw1OutVolt = kPCA9420_Sw1OutVolt0V700;

    /* Mode 2: VDDCORE off. */
    cfg[2].enableSw1Out = false;

    /* Mode 3: VDDCORE, VDD1V8 and VDDIO off. */
    cfg[3].enableSw1Out  = false;
    cfg[3].enableSw2Out  = false;
    cfg[3].enableLdo2Out = false;
}
#if POWER_DOWN_PLL_BEFORE_DEEP_SLEEP
void BOARD_DisablePll(void)
{
    /* Let FlexSPI run on FRO. */
    BOARD_SetFlexspiClock(CLKCTL0_FLEXSPIFCLKSEL_SEL(3), 1);
    /* Let CPU run on ffro before power down SYS PLL. */
    CLOCK_AttachClk(kFFRO_to_MAIN_CLK);
    /* Disable the PFD clock output first. */
    CLOCK_DeinitSysPfd(kCLOCK_Pfd0);
    CLOCK_DeinitSysPfd(kCLOCK_Pfd2);
    CLOCK_DeinitAudioPfd(kCLOCK_Pfd0);
    /* Disable PLL. */
    CLOCK_DeinitSysPll();
    CLOCK_DeinitAudioPll();
    /* Power down SysOsc */
    POWER_EnablePD(kPDRUNCFG_PD_SYSXTAL);
}

void BOARD_RestorePll(void)
{
    /* Power on SysOsc */
    POWER_DisablePD(kPDRUNCFG_PD_SYSXTAL);
    CLOCK_EnableSysOscClk(true, BOARD_SYSOSC_SETTLING_US);
    /*Restore PLL*/
    CLOCK_InitSysPll(&g_sysPllConfig_BOARD_BootClockRUN);
    CLOCK_InitAudioPll(&g_audioPllConfig_BOARD_BootClockRUN);
    /*Restore PFD*/
    CLOCK_InitSysPfd(kCLOCK_Pfd0, 19);   /* Enable main PLL clock 500MHz. */
    CLOCK_InitSysPfd(kCLOCK_Pfd2, 24);   /* Enable aux0 PLL clock 396MHz for SDIO */
    CLOCK_InitAudioPfd(kCLOCK_Pfd0, 26); /* Configure audio_pll_clk to 24.576Mhz */
    /* Let CPU run on SYS PLL PFD0 with divider 2 (250Mhz). */
    CLOCK_AttachClk(kMAIN_PLL_to_MAIN_CLK);
    /* Move FlexSPI clock source from FRO clock to Main clock. */
    BOARD_SetFlexspiClock(CLKCTL0_FLEXSPIFCLKSEL_SEL(0), 2);
}
#endif /* POWER_DOWN_PLL_BEFORE_DEEP_SLEEP */

/*!
 * @brief To Enable/Disable the clock
 */
static void APP_ConfigureClock(void)
{
	CLOCK_EnableClock(kCLOCK_OtpCtrl);
    OCOTP->OTP_PDN = OCOTP_OTP_PDN_PDN_MASK;
    CLOCK_DisableClock(kCLOCK_OtpCtrl);

    CLKCTL0->SYSTICKFCLKDIV = 0x40000000;
    CLOCK_AttachClk(kNONE_to_SYSTICK_CLK);

    CLKCTL0->WAKECLK32KHZDIV = 0x40000000;
    CLOCK_AttachClk(kNONE_to_32KHZWAKE_CLK);

    CLOCK_AttachClk(kNONE_to_CLKOUT);

    CLOCK_AttachClk(kNONE_to_USB_CLK);
    CLOCK_AttachClk(kNONE_to_SCT_CLK);

    CLOCK_DisableClock(kCLOCK_PowerQuad);
    CLOCK_DisableClock(kCLOCK_Casper);
    CLOCK_DisableClock(kCLOCK_HashCrypt);
    CLOCK_DisableClock(kCLOCK_Puf);
    CLOCK_DisableClock(kCLOCK_Rng);
    CLOCK_DisableClock(kCLOCK_OtpCtrl);
    CLOCK_DisableClock(kCLOCK_UsbhsPhy);
    CLOCK_DisableClock(kCLOCK_UsbhsDevice);
    CLOCK_DisableClock(kCLOCK_UsbhsHost);
    CLOCK_DisableClock(kCLOCK_UsbhsSram);
    CLOCK_DisableClock(kCLOCK_Sct);

    CLKCTL0->SDIO0FCLKDIV = 0x40000000;
    CLOCK_AttachClk(kNONE_to_SDIO0_CLK);
    CLKCTL0->SDIO1FCLKDIV = 0x40000000;
    CLOCK_AttachClk(kNONE_to_SDIO1_CLK);
    CLOCK_AttachClk(kNONE_to_ACMP_CLK);

    CLOCK_DisableClock(kCLOCK_Sdio0);
    CLOCK_DisableClock(kCLOCK_Sdio1);
    CLOCK_DisableClock(kCLOCK_Acmp0);
    CLOCK_DisableClock(kCLOCK_Adc0);
    CLOCK_DisableClock(kCLOCK_ShsGpio0);

    CLOCK_AttachClk(kNONE_to_UTICK_CLK);
    CLOCK_AttachClk(kNONE_to_WDT0_CLK);

    CLOCK_DisableClock(kCLOCK_Utick0);
    CLOCK_DisableClock(kCLOCK_Wwdt0);

    CLKCTL1->FLEXCOMM[1].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM1);
    CLKCTL1->FLEXCOMM[2].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM2);
    CLKCTL1->FLEXCOMM[3].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM3);
    CLKCTL1->FLEXCOMM[4].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM4);
    CLKCTL1->FLEXCOMM[5].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM5);
    CLKCTL1->FLEXCOMM[6].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM6);
    CLKCTL1->FLEXCOMM[7].FRGCLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM7);
    CLKCTL1->FRG14CLKSEL = 7;
    CLOCK_AttachClk(kNONE_to_FLEXCOMM14);
    CLOCK_AttachClk(kNONE_to_DMIC_CLK);
    CLOCK_AttachClk(kNONE_to_OSTIMER_CLK);

    CLOCK_DisableClock(kCLOCK_Flexcomm1);
    CLOCK_DisableClock(kCLOCK_Flexcomm2);
    CLOCK_DisableClock(kCLOCK_Flexcomm3);
    CLOCK_DisableClock(kCLOCK_Flexcomm4);
    CLOCK_DisableClock(kCLOCK_Flexcomm5);
    CLOCK_DisableClock(kCLOCK_Flexcomm6);
    CLOCK_DisableClock(kCLOCK_Flexcomm7);
    CLOCK_DisableClock(kCLOCK_Flexcomm14);
    CLOCK_DisableClock(kCLOCK_Dmic0);
    CLOCK_DisableClock(kCLOCK_OsEventTimer);

    CLOCK_DisableClock(kCLOCK_HsGpio2);
    CLOCK_DisableClock(kCLOCK_HsGpio3);
    CLOCK_DisableClock(kCLOCK_HsGpio4);
    CLOCK_DisableClock(kCLOCK_HsGpio5);
    CLOCK_DisableClock(kCLOCK_HsGpio6);
    CLOCK_DisableClock(kCLOCK_HsGpio7);
    CLOCK_DisableClock(kCLOCK_Crc);
    CLOCK_DisableClock(kCLOCK_Dmac0);
    CLOCK_DisableClock(kCLOCK_Dmac1);
    CLOCK_DisableClock(kCLOCK_Mu);
    CLOCK_DisableClock(kCLOCK_Sema);
    CLOCK_DisableClock(kCLOCK_Freqme);

    CLOCK_AttachClk(kNONE_to_CTIMER0);
    CLOCK_AttachClk(kNONE_to_CTIMER1);
    CLOCK_AttachClk(kNONE_to_CTIMER2);
    CLOCK_AttachClk(kNONE_to_CTIMER3);
    CLOCK_AttachClk(kNONE_to_CTIMER4);
    CLOCK_AttachClk(kNONE_to_WDT1_CLK);
    CLOCK_AttachClk(kNONE_to_I3C_CLK);
    CLOCK_AttachClk(kNONE_to_I3C_TC_CLK);

    CLOCK_DisableClock(kCLOCK_Ct32b0);
    CLOCK_DisableClock(kCLOCK_Ct32b1);
    CLOCK_DisableClock(kCLOCK_Ct32b2);
    CLOCK_DisableClock(kCLOCK_Ct32b3);
    CLOCK_DisableClock(kCLOCK_Ct32b4);
    CLOCK_DisableClock(kCLOCK_Rtc);
    CLOCK_DisableClock(kCLOCK_Mrt0);
    CLOCK_DisableClock(kCLOCK_Wwdt1);
    CLOCK_DisableClock(kCLOCK_I3c0);
}

/*!
 * @brief Enabling the bit powers down the peripheral
 */
static void APP_PowerDownPeripheral(void)
{
    POWER_EnablePD(kPDRUNCFG_PD_LPOSC);
    POWER_EnablePD(kPDRUNCFG_PD_ADC);
    POWER_EnablePD(kPDRUNCFG_LP_ADC);
    POWER_EnablePD(kPDRUNCFG_PD_ADC_TEMPSNS);
    POWER_EnablePD(kPDRUNCFG_PD_ACMP);
    POWER_EnablePD((pd_bit_t)MAKE_PD_BITS(PDRCFG0, 26U));
    POWER_EnablePD((pd_bit_t)MAKE_PD_BITS(PDRCFG0, 27U));
    POWER_EnablePD((pd_bit_t)MAKE_PD_BITS(PDRCFG0, 28U));
    POWER_EnablePD((pd_bit_t)MAKE_PD_BITS(PDRCFG0, 29U));

    POWER_EnablePD(kPDRUNCFG_APD_PQ_SRAM);
    POWER_EnablePD(kPDRUNCFG_PPD_PQ_SRAM);
    POWER_EnablePD(kPDRUNCFG_APD_USBHS_SRAM);
    POWER_EnablePD(kPDRUNCFG_PPD_USBHS_SRAM);
    POWER_EnablePD(kPDRUNCFG_APD_USDHC0_SRAM);
    POWER_EnablePD(kPDRUNCFG_PPD_USDHC0_SRAM);
    POWER_EnablePD(kPDRUNCFG_APD_USDHC1_SRAM);
    POWER_EnablePD(kPDRUNCFG_PPD_USDHC1_SRAM);
    POWER_EnablePD(kPDRUNCFG_APD_CASPER_SRAM);
    POWER_EnablePD(kPDRUNCFG_PPD_CASPER_SRAM);

    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF0);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF1);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF2);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF3);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF4);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF5);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF6);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF7);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF8);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF9);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF10);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF11);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF14);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF15);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF16);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF17);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF18);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF19);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF20);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF21);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF22);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF23);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF24);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF25);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF26);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF27);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF28);
    POWER_EnablePD(kPDRUNCFG_APD_SRAM_IF29);

    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF0);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF1);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF2);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF3);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF4);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF5);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF6);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF7);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF8);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF9);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF10);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF11);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF14);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF15);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF16);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF17);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF18);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF19);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF20);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF21);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF22);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF23);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF24);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF25);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF26);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF27);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF28);
    POWER_EnablePD(kPDRUNCFG_PPD_SRAM_IF29);
}

/*!
 * @brief Main function
 */
int main(void)
{
    /* Init board hardware. */
    pca9420_modecfg_t pca9420ModeCfg[4];
    uint32_t i;
#if SLEEP_AT_48M_IRC_DIV4
    uint32_t cpu_div;
    uint32_t mainclk_sel[2];
    uint32_t dspclk_sel[2];
#endif

    /* BE CAUTIOUS TO SET CORRECT VOLTAGE RANGE ACCORDING TO YOUR BOARD/APPLICATION. PAD SUPPLY BEYOND THE RANGE DO
       HARM TO THE SILICON. */
    power_pad_vrange_t vrange = {.Vdde0Range = kPadVol_171_198,
                                 .Vdde1Range = kPadVol_171_360,
                                 /* SD0 voltage is switchable, but in power_manager demo, it's fixed 3.3V. */
                                 .Vdde2Range = kPadVol_300_360};

    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    APP_ConfigureClock();
    APP_PowerDownPeripheral();

    POWER_ApplyPD();

    /* PMIC PCA9420 */
    BOARD_InitPmic();
    for (i = 0; i < ARRAY_SIZE(pca9420ModeCfg); i++)
    {
        PCA9420_GetDefaultModeConfig(&pca9420ModeCfg[i]);
    }
    BOARD_ConfigPMICModes(pca9420ModeCfg, ARRAY_SIZE(pca9420ModeCfg));
    PCA9420_WriteModeConfigs(&pca9420Handle, kPCA9420_Mode0, &pca9420ModeCfg[0], ARRAY_SIZE(pca9420ModeCfg));

    /* Configure PMIC Vddcore value according to main clock. DSP not used in this demo. */
    BOARD_SetPmicVoltageForFreq(kPartTemp_N20C_P70C, CLOCK_GetMainClkFreq(), 0U);

    /* Indicate to power library that PMIC is used. */
    POWER_UpdatePmicRecoveryTime(1);
    
    POWER_SetPadVolRange(&vrange);

    /* Determine the power mode before bring up. */
    if ((POWER_GetEventFlags() & PMC_FLAGS_DEEPPDF_MASK) != 0)
    {
        PRINTF("Board wake up from deep or full deep power down mode.\r\n");
        POWER_ClearEventFlags(PMC_FLAGS_DEEPPDF_MASK);
    }

    APP_InitWakeupPin();
    EnableDeepSleepIRQ(PMC_PMIC_IRQn);
    GPIO_PortInit(GPIO, 0);
    GPIO_PinInit(GPIO, 0, 31, &(gpio_pin_config_t){kGPIO_DigitalOutput, 1});

    PRINTF("Power Manager Demo.\r\n");
    PRINTF("The \"user key\" is: %s\r\n", APP_USER_WAKEUP_KEY_NAME);

    while (1)
    {
        gCurrentPowerMode = APP_GetUserSelection();
        PRINTF("Entering %s ...\r\n", gWakeupInfoStr[gCurrentPowerMode]);
        pintFlag = false;
        /* Enter the low power mode. */
        switch (gCurrentPowerMode)
        {
            case kPmu_Sleep: /* Enter sleep mode. */
#if SLEEP_AT_48M_IRC_DIV4
                mainclk_sel[0] = CLKCTL0->MAINCLKSELA;
                mainclk_sel[1] = CLKCTL0->MAINCLKSELB;
                dspclk_sel[0]  = CLKCTL1->DSPCPUCLKSELA;
                dspclk_sel[1]  = CLKCTL1->DSPCPUCLKSELB;
                cpu_div        = CLKCTL0->SYSCPUAHBCLKDIV;
                CLKCTL0->MAINCLKSELA     = CLKCTL0_MAINCLKSELA_SEL(0);
                CLKCTL0->MAINCLKSELB     = CLKCTL0_MAINCLKSELB_SEL(0);
                CLKCTL1->DSPCPUCLKSELA   = CLKCTL1_DSPCPUCLKSELA_SEL(0);
                CLKCTL1->DSPCPUCLKSELB   = CLKCTL1_DSPCPUCLKSELB_SEL(0);
                CLKCTL0->SYSCPUAHBCLKDIV = 0;
                while (CLKCTL0->SYSCPUAHBCLKDIV & CLKCTL0_SYSCPUAHBCLKDIV_REQFLAG_MASK)
                {
                }
#endif
                POWER_EnterSleep();
#if SLEEP_AT_48M_IRC_DIV4
                CLKCTL0->SYSCPUAHBCLKDIV = cpu_div;
                while (CLKCTL0->SYSCPUAHBCLKDIV & CLKCTL0_SYSCPUAHBCLKDIV_REQFLAG_MASK)
                {
                }
                CLKCTL0->MAINCLKSELA   = mainclk_sel[0] & CLKCTL0_MAINCLKSELA_SEL_MASK;
                CLKCTL0->MAINCLKSELB   = mainclk_sel[1] & CLKCTL0_MAINCLKSELB_SEL_MASK;
                CLKCTL1->DSPCPUCLKSELA = dspclk_sel[0] & CLKCTL1_DSPCPUCLKSELA_SEL_MASK;
                CLKCTL1->DSPCPUCLKSELB = dspclk_sel[1] & CLKCTL1_DSPCPUCLKSELB_SEL_MASK;
#endif
                break;
            case kPmu_Deep_Sleep: /* Enter deep sleep mode. */
                BOARD_SetPmicVoltageBeforeDeepSleep();
#if POWER_DOWN_PLL_BEFORE_DEEP_SLEEP
                /* Disable Pll before enter deep sleep mode */
                BOARD_DisablePll();
#endif
                POWER_EnterDeepSleep(APP_EXCLUDE_FROM_DEEPSLEEP);
#if POWER_DOWN_PLL_BEFORE_DEEP_SLEEP
                /* Restore Pll before enter deep sleep mode */
                BOARD_RestorePll();
#endif
                BOARD_RestorePmicVoltageAfterDeepSleep();
                break;
            case kPmu_Deep_PowerDown: /* Enter deep power down mode. */
                PRINTF(
                    "Press any key to confirm to enter the deep power down mode and wakeup the device by "
                    "reset.\r\n\r\n");
                GETCHAR();
#if WAKE_UP_WITH_PMIC_IRQ_N
                POWER_EnableInterrupts(PMC_CTRL_INTRPADEN_MASK);
#endif
                POWER_EnterDeepPowerDown(APP_EXCLUDE_FROM_DEEP_POWERDOWN);
                break;
            case kPmu_Full_Deep_PowerDown: /* Enter full deep power down mode. */
                PRINTF(
                    "Press any key to confirm to enter the full deep power down mode and wakeup the device by "
                    "reset.\r\n\r\n");
                GETCHAR();
#if WAKE_UP_WITH_PMIC_IRQ_N
                POWER_EnableInterrupts(PMC_CTRL_INTRPADEN_MASK);
#endif
                POWER_EnterFullDeepPowerDown(APP_EXCLUDE_FROM_FULL_DEEP_POWERDOWN);
                break;
            default:
                break;
        }
        if (pintFlag)
        {
            PRINTF("Pin event occurs\r\n");
        }
        PRINTF("Wakeup.\r\n");
    }
}

/*
 * Setup a GPIO input pin as wakeup source.
 */
static void APP_InitWakeupPin(void)
{
    gpio_pin_config_t gpioPinConfigStruct;

    /* Set SW pin as GPIO input. */
    gpioPinConfigStruct.pinDirection = kGPIO_DigitalInput;
    GPIO_PinInit(APP_USER_WAKEUP_KEY_GPIO, APP_USER_WAKEUP_KEY_PORT, APP_USER_WAKEUP_KEY_PIN, &gpioPinConfigStruct);

    /* Configure the Input Mux block and connect the trigger source to PinInt channle. */
    INPUTMUX_Init(INPUTMUX);
    INPUTMUX_AttachSignal(INPUTMUX, kPINT_PinInt0, APP_USER_WAKEUP_KEY_INPUTMUX_SEL); /* Using channel 0. */
    INPUTMUX_Deinit(INPUTMUX); /* Turnoff clock to inputmux to save power. Clock is only needed to make changes */

    /* Configure the interrupt for SW pin. */
    PINT_Init(PINT);
    PINT_PinInterruptConfig(PINT, kPINT_PinInt0, kPINT_PinIntEnableFallEdge, pint_intr_callback);
    PINT_EnableCallback(PINT); /* Enable callbacks for PINT */

    EnableDeepSleepIRQ(PIN_INT0_IRQn);
}

/*
 * Callback function when wakeup key is pressed.
 */
static void pint_intr_callback(pint_pin_int_t pintr, uint32_t pmatch_status)
{
    pintFlag = true;
}

/*
 * Get user selection from UART.
 */
static uint32_t APP_GetUserSelection(void)
{
    uint32_t ch;

    /* Clear rx overflow error once it happens during low power mode. */
    if (APP_USART_RX_ERROR == (APP_USART_RX_ERROR & USART_GetStatusFlags((USART_Type *)BOARD_DEBUG_UART_BASEADDR)))
    {
        USART_ClearStatusFlags((USART_Type *)BOARD_DEBUG_UART_BASEADDR, APP_USART_RX_ERROR);
    }

    PRINTF(
        "Select an option\r\n"
        "\t1. Sleep mode\r\n"
        "\t2. Deep Sleep mode\r\n"
        "\t3. Deep power down mode\r\n"
        "\t4. Full deep power down mode\r\n");
    while (1)
    {
        ch = GETCHAR();
        if ((ch < '1') || (ch > '4')) /* Only '1', '2', '3' , '4'. */
        {
            continue;
        }
        else
        {
            ch = ch - '1'; /* Only 0, 1, 2 , 3 . */
            break;
        }
    }
    switch (ch)
    {
        case 0:
            ch = kPmu_Sleep;
            break;
        case 1:
            ch = kPmu_Deep_Sleep;
            break;
        case 2:
            ch = kPmu_Deep_PowerDown;
            break;
        case 3:
            ch = kPmu_Full_Deep_PowerDown;
            break;
        default:
            break;
    }
    return ch;
}
