/*********************************Header File********************************************************/
#include "BLDC_sct.h"
#include "mlib.h"
#include "mlib_types.h"
#include "fsl_gpio.h"
#include "fsl_inputmux.h"
#include "fsl_pint.h"
/*********************************Header File********************************************************/

/*********************************Macro Definition***************************************************/
//SCT only match event macro definition
#define EVCTR_MATCH_EVENT(x)    (SCT_EV_CTRL_COMBMODE(1) | SCT_EV_CTRL_MATCHSEL(x) )
//SCT only IO event macro definition
#define EVCTR_IN_HIGH_EVENT(x)  (SCT_EV_CTRL_COMBMODE(2) | SCT_EV_CTRL_OUTSEL(0) | SCT_EV_CTRL_IOSEL(x) | SCT_EV_CTRL_IOCOND(0))
//SCT state set macro
#define	EVCTR_STATE_SET(x)      (SCT_EV_CTRL_STATELD(1)  | SCT_EV_CTRL_STATEV(x))
//SCT state no change macro
#define	EVCTR_STATE_NO_CHANGE	(SCT_EV_CTRL_STATELD(0)  | SCT_EV_CTRL_STATEV(0))

//ADC trigger port
#define SCT_ADC_TRIGG_PORT      (1u)
//ADC trigger pin
#define SCT_ADC_TRIGG_PIN       (19u)

/*
 * PT and PB are SCT output channels which complementary PWM pair are applied on
 * H is SCT output channel which high level is applied on
 * L2, L1 and L0 are SCT output channels which low level are applied on
*/
#define PWM_TB_H_Lx(PT, PB, H, L2, L1, L0) (PT<<28 | PB<<24 | H<<12 | L2<<8 | L1<<4 | L0<<0)

// free-running timer prescaler requirements: to be used with 1 Hz - 100 Hz (60 - 6000 rpm)
// @30 MHz - 30*10^6 / 1 /6 = 5*10^6 counts/Hall edge => prescaler of at least 77 so that the counter stays 16 bits
//           30*10^6 / 100 /6 = 50000 counts/Hall edge => prescaler not needed
#define SCT_SPEED_PRESC    (11u)
/*********************************Macro Definition***************************************************/

/*********************************Variable Definition************************************************/
extern   uint32_t          current_seqa_ctrl;
extern   volatile int      Uadc, Iadc;
extern   volatile float    Uv, Ia;
extern   frac16_t          f16CaptureSpeed,f16SpeedCoef;
volatile unsigned char     g_InitPosVal = 0;
volatile unsigned char     g_CurPosition = 0;
volatile unsigned char     g_LstPosition = 0;
extern   int16_t           cap_cnt;
extern   int16_t           cap_cntFilt;
         frac16_t          f16CaptureFilterArr[16];
         uint8_t           ui8CaptureFilterIndex;
volatile uint32_t          direction_estimate;
volatile uint32_t          speed_timer_rd_new;
volatile uint32_t          step_buffer_direction;
volatile uint32_t          hall_edges_delta, hall_edges_delta_valid;
volatile uint16_t          pwm_top_active = 1000;
volatile uint16_t          pwm_xy_deadtime = 12;
         uint8_t           s_Mdirection = 0;
extern   volatile uint32_t speed_limit_monitor;
volatile uint32_t          speed_limit_monitor_old;
// this array specifies pwm output based on the hall reading and selected direction (0-cw, 1-ccw)
// hall sensor A is encoded as MSB, C as LSB; hall sensor readings are used as index
// sub-array 0: specifies clockwise steps
// sub-array 1: specifies couter clockwise steps
//PWM control
const unsigned long hall_to_pwm[2][8] = 
{
  { 
    0,                        //000
    PWM_TB_H_Lx(5,6,4,3,2,1), //001 
    PWM_TB_H_Lx(3,4,2,6,5,1), //010
    PWM_TB_H_Lx(5,6,2,4,3,1), //011
    PWM_TB_H_Lx(1,2,6,5,4,3), //100
    PWM_TB_H_Lx(1,2,4,6,5,3), //101
    PWM_TB_H_Lx(3,4,6,5,2,1), //110
    0                         //111
  }, 
  { 0,                        //000
    PWM_TB_H_Lx(3,4,6,5,2,1), //001 
    PWM_TB_H_Lx(1,2,4,6,5,3), //010
    PWM_TB_H_Lx(1,2,6,5,4,3), //011
    PWM_TB_H_Lx(5,6,2,4,3,1), //100
    PWM_TB_H_Lx(3,4,2,6,5,1), //101
    PWM_TB_H_Lx(5,6,4,3,2,1), //110
    0                         //111
  }
};

//                                                1  2  3  4  5  6
const unsigned long out_pwm1_6_2_sct_out[7] = {0, 0, 1, 2, 3, 5, 6};
/*********************************Variable Definition************************************************/

/*********************************Type Definition****************************************************/
//SCT IO signal
enum SCT0_in	
{
  app_in_hsensors = 0,     //Input signal related event 0
  app_in_debug_halt = 1   //Input signal related event 1
};

//SCT PWM output channel number
enum SCT0_out
{
  app_out_pwm1 = 0,       //PWM out 1   is from SCT PWM output 0 channel
  app_out_pwm2 = 1,       //PWM out 2   is from SCT PWM output 1 channel
  app_out_pwm3 = 2,       //PWM out 3   is from SCT PWM output 2 channel
  app_out_pwm4 = 3,       //PWM out 4   is from SCT PWM output 3 channel
  app_out_pwm5 = 5,       //PWM out 5   is from SCT PWM output 5 channel
  app_out_pwm6 = 6,       //PWM out 6   is from SCT PWM output 6 channel
  app_out_adc_trigger = 7, //ADC trigger is from SCT PWM output 7 channel
  app_out_debug_halt = 8 //app out when debug halt
};

//SCT match register number
enum SCT0_pwm_match 
{
  app_pwm_m_update0 = 0,  //event 1 <-> match 0 register
  app_pwm_m_update1,      //event 2 <-> match 1 register
  app_pwm_m_edge0,        //event 3 <-> match 2 register
  app_pwm_m_edge1,        //event 4 <-> match 3 register
  app_pwm_m_edge2,        //event 5 <-> match 4 register
  app_pwm_m_edge3,        //event 6 <-> match 5 register
  app_pwm_m_adc_trigger   //event 7 <-> match 6 register
};

//SCT event definition
enum SCT0_event 
{
  app_pwm_ev_hsensor_edge = 0, //event 0->check Hall sensor edge
  app_pwm_ev_update0,          //event 1->set outputs to 0
  app_pwm_ev_update1,          //event 2->set outputs to 1 and prepare PWM outputs
  app_pwm_ev_edge0,            //event 3->generate edge 0
  app_pwm_ev_edge1,            //event 4->generate edge 1
  app_pwm_ev_edge2,            //event 5->generate edge 2
  app_pwm_ev_edge3,            //event 6->generate edge 3
  app_pwm_ev_adc_trigger       //event 7->generate adc trigger
};

//SCT state machine
enum SCT0_pwm_state 
{
  app_pwm_st_idle = 0, //This is default state of SCTimer state machine
  app_pwm_st_active,   //This state is entered to generate the complementary PWM waveform 
  app_pwm_st_transient //This state is entered when the pattern-matching engine(PME) 
                       //generates a falling edge and that happens when it detects an edge 
                       //at one of the Hall sensors. This state is used to prepare 
                       //six SCT outputs that drive the motor for the next 1/6 of a revolution.
};
/*********************************Type Definition****************************************************/

/*********************************Function Definition************************************************/
/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: GET_MPOS
*
* @brief   The function gets the motor position, reading the hall-sensors. 
*          The function returns the motor position.
*        
* @param   none
*
* @return s_MotorPosition
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
unsigned char GET_MPOS(void) 
{
  //BLDC rotor position variable
  unsigned char s_MotorPosition;  
  //Count variable
  unsigned int  s_i;
  //Hall sensor phase A status
  uint32_t      pha;
  //Hall sensor phase B status
  uint32_t      phb;
  //Hall sensor phase C status
  uint32_t      phc;
  
  //a short delay
  for(s_i=0; s_i<10; s_i++)
  {
  };
  //clear BLDC rotor position variable
  s_MotorPosition = 0;
  //Get Hall sensor A phase status
  pha = ((GPIO->PIN[0] & (1 << M_PHA)) >> M_PHA) << 2;
  //Get Hall sensor B phase status
  phb = ((GPIO->PIN[0] & (1 << M_PHB)) >> M_PHB) << 1;
  //Get Hall sensor C phase status
  phc = ((GPIO->PIN[0] & (1 << M_PHC)) >> M_PHC) << 0;
  //Get BLDC rotor position 
  s_MotorPosition = pha | phb | phc;
  
  return s_MotorPosition;
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: invalidate_period_estimate
*
* @brief   invalidate period estimate
*        
* @param   void
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void invalidate_period_estimate(void)
{
  step_buffer_direction = 2;
  hall_edges_delta_valid = 0;

  return;
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: update_sct_out
*
* @brief    The function is used to update SCTimer output channels while motor commutates between six states.
*        
* @param    index
*
* @return  none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//This is a redesigned routine based on the correct cw/ccw sequence !!!
void update_sct_out(unsigned long pwm_control)
{
  //SCTimer output channels for outputing complementary PWMs to top MOS and bottom MOS
  unsigned long pwm_out_t, pwm_out_b;
  //SCTimer output channel for outputing high level to MOS
  unsigned long pwm_out_h;
  //SCTimer output channels for outputing low level to MOS
  unsigned long pwm_out_l2, pwm_out_l1, pwm_out_l0;
  
  /******************************1. Get SCTimer output channel for outputing complementary PWMs, high level, low level*****************/
  pwm_out_t  = out_pwm1_6_2_sct_out[(pwm_control>>28) & 0x0F];
  pwm_out_b  = out_pwm1_6_2_sct_out[(pwm_control>>24) & 0x0F];
  pwm_out_h  = out_pwm1_6_2_sct_out[(pwm_control>>12) & 0x0F];
  pwm_out_l2 = out_pwm1_6_2_sct_out[(pwm_control>>8 ) & 0x0F];
  pwm_out_l1 = out_pwm1_6_2_sct_out[(pwm_control>>4 ) & 0x0F];
  pwm_out_l0 = out_pwm1_6_2_sct_out[(pwm_control>>0 ) & 0x0F];
  /******************************1. Get SCTimer output channel for outputing complementary PWMs, high level, low level*****************/
  
  /******************************2. Update SCTimer output channels*********************************************************************/   
  SCT0->OUT[pwm_out_t].SET = 1<<app_pwm_ev_edge0;
  SCT0->OUT[pwm_out_t].CLR = 1<<app_pwm_ev_edge1 | 1<<app_pwm_ev_update0;
	
  SCT0->OUT[pwm_out_b].SET = 1<<app_pwm_ev_edge2;
  SCT0->OUT[pwm_out_b].CLR = 1<<app_pwm_ev_edge3 | 1<<app_pwm_ev_update0;
	
  SCT0->OUT[pwm_out_h].SET = 1<<app_pwm_ev_update1;
  SCT0->OUT[pwm_out_h].CLR = 0;
	
  SCT0->OUT[pwm_out_l2].SET = 0;
  SCT0->OUT[pwm_out_l2].CLR = 1<<app_pwm_ev_update0;

  SCT0->OUT[pwm_out_l1].SET = 0;
  SCT0->OUT[pwm_out_l1].CLR = 1<<app_pwm_ev_update0;

  SCT0->OUT[pwm_out_l0].SET = 0;
  SCT0->OUT[pwm_out_l0].CLR = 1<<app_pwm_ev_update0;
  /******************************2. Update SCTimer output channels*********************************************************************/   
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: SCT0_IRQHandler
*
* @brief   The function is used to handle SCT interrupt for detecting new position and current motor speed.
*        
* @param   none
*
* @return  status
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
int SCT0_IRQHandler(void)
{
  //debug purpose 
  speed_limit_monitor_old = speed_limit_monitor; 
  //clear speed_limit_monitor used to check if motor running
  speed_limit_monitor =0;  
  
  /******************capture free running SCT counter value**************************************/
  //capture free running SCT counter value
  speed_timer_rd_new  = (SCT0->COUNT & SCT_COUNT_CTR_H_MASK)>>SCT_COUNT_CTR_H_SHIFT;
  /*
   * The reading times of the SCT count value register is limited to 7 times, 
   * and it is cleared more than 7 times
  */
  if (ui8CaptureFilterIndex >= 7)
  {
    ui8CaptureFilterIndex = 0;
  }
  //save SCT count value to capture filter array
  f16CaptureFilterArr[ui8CaptureFilterIndex] = speed_timer_rd_new;
  //capture filter array index increment 1
  ui8CaptureFilterIndex++;
  //SCT count value captured
  cap_cntFilt =  speed_timer_rd_new;/*   (  f16CaptureFilterArr[0] 
                       + f16CaptureFilterArr[1] 
                       + f16CaptureFilterArr[2] 
                       + f16CaptureFilterArr[3] 
                       + f16CaptureFilterArr[4] 
                       + f16CaptureFilterArr[5]
                       + f16CaptureFilterArr[6] 
                       + f16CaptureFilterArr[7])>>3;*/
  /******************capture free running SCT counter value**************************************/
  
  /******************Clear any pending or left-over edge-detect logic****************************/
  PINT->IST = 0x07<<0;
  PINT->RISE = 0x07<<0;
  PINT->FALL = 0x07<<0;
  /******************Clear any pending or left-over edge-detect logic****************************/
  /******************Configure pattern match interrupt control register**************************/
  //1.Interrupts are driven in response to pattern matches
  //2.RXEV output to the CPU is enabled
  PINT->PMCTRL = 0;
  PINT->PMCTRL = 3;
  /******************Configure pattern match interrupt control register**************************/
  
  /******************Clear Event 0 flag**********************************************************/
  //Event 0 is used to check Hall sensor edge
  SCT0->EVFLAG = 1<<app_pwm_ev_hsensor_edge;
  /******************Clear Event 0 flag**********************************************************/
  
  /******************BLDC motor position detection and update SCTimer PWM outputs****************/
  //Get BLDC motor position
  g_CurPosition = GET_MPOS();
  //Detect whether BLDC needs commutation or not
  if (g_CurPosition!=g_LstPosition)
  {  
    /*
     * update the SCTimer PWM outputs according to the BLDC motor rotor position 
     * and rotation direction combined with the commutation table
    */    
    update_sct_out(hall_to_pwm[s_Mdirection][g_CurPosition]);
  }
  //After the SCT PWM outputs are updated, the current position becomes the previous position
  g_LstPosition = g_CurPosition;
  /******************BLDC motor position detection and update SCTimer PWM outputs****************/
  
  //PWM reload register update begin
  GPIO->SET[SCT_ADC_TRIGG_PORT] = 1<<SCT_ADC_TRIGG_PIN;
  GPIO->CLR[SCT_ADC_TRIGG_PORT] = 1<<SCT_ADC_TRIGG_PIN;
  //PWM reload register update end
  
  //halt upper part of SCT
  SCT0->CTRL |= SCT_CTRL_HALT_H(1); 
  //reset counter value 
  SCT0->CTRL |= SCT_CTRL_CLRCTR_H_MASK; 
     
  //Make SCTimer low part and high part run
  SCT0->CTRL &= ~(SCT_CTRL_STOP_H(1) | SCT_CTRL_HALT_H(1) | SCT_CTRL_STOP_L(1) | SCT_CTRL_HALT_L(1));
  
  return true;
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: Inputmux_Init
*
* @brief   The function is used to initialize inputmux
*        
* @param   none
*
* @return  none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void Inputmux_Init(void)
{
  /******************Enable Input Mux clock and Reset Input Mux************************************************/
  //Enable the INPUTMUX clock
  INPUTMUX_Init(INPUTMUX);
  //Reset INPUTMUX including assert reset and then clear reset
  RESET_PeripheralReset(kMUX_RST_SHIFT_RSTn);
  /******************Enable Input Mux clock and Reset Input Mux************************************************/
  
  /******************Select pin for pattern match input 0,1,2**************************************************/
  INPUTMUX_AttachSignal(INPUTMUX, kPINT_PatternMatchInp0Src, kINPUTMUX_GpioPort0Pin2ToPintsel);
  INPUTMUX_AttachSignal(INPUTMUX, kPINT_PatternMatchInp1Src, kINPUTMUX_GpioPort0Pin3ToPintsel);
  INPUTMUX_AttachSignal(INPUTMUX, kPINT_PatternMatchInp2Src, kINPUTMUX_GpioPort0Pin5ToPintsel);
  /******************Select pin for pattern match input 0,1,2**************************************************/
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: pattern_match_callback
*
* @brief   The function is used as pattern match callback function
*        
* @param   none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void pattern_match_callback(pint_pin_int_t pintr, uint32_t pmatch_status)
{
  /**********************generate edges as SCT interrupt source*************************/
  //drive pin 6 on port 0 to low
  GPIO_PinWrite(GPIO, 0, 6, 0);
  //a short delay
  __NOP();
  //drive pin 6 on port 0 to high
  GPIO_PinWrite(GPIO, 0, 6, 1);
  /**********************generate edges as SCT interrupt source*************************/
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: PME_Init
*
* @brief   The function is used to initialize pattern match engine
*        
* @param   none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void PME_Init(void)
{
  //pint and pmatch configuration structure
  pint_pmatch_cfg_t pmcfg;
  
  /******************Configure Pattern Match Engine***********************************************************/
  //Initialize PINT
  PINT_Init(PINT);
  
  //Setup Pattern Match Bit Slice 0
  pmcfg.bs_src    = kPINT_PatternMatchInp0Src;
  pmcfg.bs_cfg    = kPINT_PatternMatchStickyBothEdges;
  pmcfg.callback  = pattern_match_callback;
  pmcfg.end_point = true;
  PINT_PatternMatchConfig(PINT, kPINT_PatternMatchBSlice0, &pmcfg);
  //Setup Pattern Match Bit Slice 1
  pmcfg.bs_src    = kPINT_PatternMatchInp1Src;
  pmcfg.bs_cfg    = kPINT_PatternMatchStickyBothEdges;
  pmcfg.callback  = pattern_match_callback;
  pmcfg.end_point = true;
  PINT_PatternMatchConfig(PINT, kPINT_PatternMatchBSlice1, &pmcfg);
  //Setup Pattern Match Bit Slice 2
  pmcfg.bs_src    = kPINT_PatternMatchInp2Src;
  pmcfg.bs_cfg    = kPINT_PatternMatchStickyBothEdges;
  pmcfg.callback  = pattern_match_callback;
  pmcfg.end_point = true;
  PINT_PatternMatchConfig(PINT, kPINT_PatternMatchBSlice2, &pmcfg);
  
  //Enable callbacks for PINT0 by Index 
  PINT_EnableCallbackByIndex(PINT, kPINT_PinInt0);
  //Enable callbacks for PINT1 by Index 
  PINT_EnableCallbackByIndex(PINT, kPINT_PinInt1);
  //Enable callbacks for PINT2 by Index 
  PINT_EnableCallbackByIndex(PINT, kPINT_PinInt2);
  
  //Enable PatternMatch
  PINT_PatternMatchEnable(PINT);
  /******************Configure Pattern Match Engine***********************************************************/
  
  /******************clear any pending or left-over edge-detect logic****************************************/
  //Clear any pending or left-over edge-detect logic
  PINT_PinInterruptClrStatusAll(PINT);
  //Clear all pin interrupt rise flags
  PINT_PinInterruptClrRiseFlagAll(PINT);
  //Clear all pin interrupt fall flags
  PINT_PinInterruptClrFallFlagAll(PINT);
  /******************clear any pending or left-over edge-detect logic****************************************/
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: SCT_Init
*
* @brief   The function is used to initiate SCT for motor control.
*        
* @param   none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void SCT_Init(void)
{
  /*********************Enable clock for SCTimer and Reset SCTimer********************************************/
  //Enable clock for SCTimer
  CLOCK_EnableClock(kCLOCK_Sct0);
  //Reset SCTimer including assert reset to SCTimer and then clear reset to SCTimer
  RESET_PeripheralReset(kSCT0_RST_SHIFT_RSTn);  
  /*********************Enable clock for SCTimer and Reset SCTimer********************************************/
  
  /*********************SCT configuration register (CONFIG)***************************************************/
  SCT0->CONFIG &= ~(SCT_CONFIG_CLKMODE(1)| SCT_CONFIG_AUTOLIMIT_L(1)  | SCT_CONFIG_AUTOLIMIT_H(1)  | SCT_CONFIG_INSYNC(1)); 
  SCT0->CONFIG |= SCT_CONFIG_CLKMODE(0)| SCT_CONFIG_AUTOLIMIT_L(0)  | SCT_CONFIG_AUTOLIMIT_H(0)  | SCT_CONFIG_INSYNC(1) ; /* SPLIT, Synchronization for input 0 */
  /*********************SCT configuration register (CONFIG)***************************************************/      
  
  /*********************SCTimer/PWM control register (CTRL)***************************************************/
  //L: no prescaler, HALT; H: prescaler, HALT
  SCT0->CTRL = SCT_CTRL_PRE_L(1-1) | SCT_CTRL_HALT_L(1)|
               SCT_CTRL_PRE_H(SCT_SPEED_PRESC-1) | SCT_CTRL_HALT_H(1);
  /*********************SCTimer/PWM control register (CTRL)***************************************************/
  
  /*********************Disable all interrupts and clear all interrupt flags**********************************/
  //disable all interrupts
  SCT0->EVEN = 0;	
  //clear all flags
  SCT0->EVFLAG = 0xFFFFFFFF;	 
  /*********************Disable all interrupts and clear all interrupt flags**********************************/
  
  /*********************SCTimer/PWM match/capture mode register (REGMODE)************************************/
  //register operates as match register
  SCT0->REGMODE = SCT_REGMODE_REGMOD_L(0) | SCT_REGMODE_REGMOD_H(0);
  /*********************SCTimer/PWM match/capture mode register (REGMODE)************************************/
  
  /******************************SCTimer match register setup************************************************/
  //update0...
  SCT0->MATCH[0]    = SCT_MATCH_MATCHn_L(49);
  SCT0->MATCHREL[0] = SCT_MATCH_MATCHn_L(49);
  
  //update1...
  SCT0->MATCH[1]    = SCT_MATCH_MATCHn_L(99);
  SCT0->MATCHREL[1] = SCT_MATCH_MATCHn_L(99);
  
  //app_pwm_m_edge0...
  SCT0->MATCH[2] =  SCT_MATCH_MATCHn_L( pwm_xy_deadtime-1 );   /* CONSTANT DEAD TIME!  */
  SCT0->MATCHREL[2] = SCT_MATCH_MATCHn_L( pwm_xy_deadtime-1 ); /* CONSTANT DEAD TIME!  */
  
  //app_pwm_m_edge1...
  SCT0->MATCH[3] = SCT_MATCH_MATCHn_L(0 );
  SCT0->MATCHREL[3] = SCT_MATCH_MATCHn_L( 0 );
  
  //app_pwm_m_edge2...
  SCT0->MATCH[4] = SCT_MATCH_MATCHn_L( 0);
  SCT0->MATCHREL[4] = SCT_MATCH_MATCHn_L( 0 );
  
  //app_pwm_m_edge3...
  SCT0->MATCH[5] = SCT_MATCH_MATCHn_L( 2048-1 );    /*Constant PWM frequency */
  SCT0->MATCHREL[5] = SCT_MATCH_MATCHn_L( 2048-1 ); /*Constatn PWM frequency */
  
  //app_pwm_m_adc_trigger...
  SCT0->MATCH[6] = SCT_MATCH_MATCHn_L( 1000 );
  SCT0->MATCHREL[6] = SCT_MATCH_MATCHn_L( 1000 );
  /******************************SCTimer match register setup************************************************/
  
  //pwm_xy_deadtime = MATCH_PWM_EDGE0;
  //pwm_x_active = MATCH_PWM_EDGE1 - MATCH_PWM_EDGE0;
  //pwm_x_active = MATCH_PWM_EDGE3 - MATCH_PWM_EDGE2;
  //pwm_xy_update_req = 0;
  //pwm_xy_update_counter = 0;
  
  /******************************SCTimer event setup*******************************************************/
  //setup for event: 0
  //enabled in state: app_pwm_st_idle and app_pwm_st_active
  //generated by: HIGH(app_in_hsensors)
  //state machine: go to state app_pwm_st_transient
  //misc control: limiting, stop, interrupt
  SCT0->EV[0].STATE = 1<<app_pwm_st_idle | 1<<app_pwm_st_active;
  SCT0->EV[0].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_IN_HIGH_EVENT(app_in_hsensors) | EVCTR_STATE_SET(app_pwm_st_transient);
  SCT0->LIMIT |= 1<<0;
  SCT0->STOP |= 1<<0;
  SCT0->EVEN |= 1<<0;
  
  //setup for event: 1
  //enabled in state: app_pwm_st_transient
  //generated by: MATCH(app_pwm_m_update0)
  //state machine: do not change state
  //output control: updated from the isr based on the rotor position and direction
  SCT0->EV[1].STATE = 1<<app_pwm_st_transient;
  SCT0->EV[1].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_update0) | EVCTR_STATE_NO_CHANGE;
  
  //setup for event: 2
  //enabled in state: app_pwm_st_transient
  //generated by: MATCH(app_pwm_m_update1)
  //state machine: go to state app_pwm_st_active
  //misc control: limiting
  //output control: updated from the isr based on the rotor position and direction
  SCT0->EV[2].STATE = 1<<app_pwm_st_transient;
  SCT0->EV[2].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_update1) | EVCTR_STATE_SET(app_pwm_st_active);
  SCT0->LIMIT |= 1<<2;
  
  //setup for event: 3
  //enabled in state: app_pwm_st_active
  //generated by: MATCH(app_pwm_m_edge0)
  //state machine: do not change state
  //output control: updated from the isr based on the rotor position and direction
  SCT0->EV[3].STATE = 1<<app_pwm_st_active;
  SCT0->EV[3].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_edge0) | EVCTR_STATE_NO_CHANGE;
  
  
  //setup for event: app_pwm_ev_edge1
  //enabled in state: app_pwm_st_active
  //generated by: MATCH(app_pwm_m_edge1)
  //state machine: do not change state
  //output control: updated from the isr based on the rotor position and direction
  SCT0->EV[4].STATE = 1<<app_pwm_st_active;
  SCT0->EV[4].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_edge1) | EVCTR_STATE_NO_CHANGE;
  
  //setup for event: app_pwm_ev_edge2
  //enabled in state: app_pwm_st_active
  //generated by: MATCH(app_pwm_m_edge2)
  //state machine: do not change state
  //output control: updated from the isr based on the rotor position and direction
  SCT0->EV[5].STATE = 1<<app_pwm_st_active;
  SCT0->EV[5].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_edge2) | EVCTR_STATE_NO_CHANGE;
  
  
  //setup for event: app_pwm_ev_edge3
  //enabled in state: app_pwm_st_active
  //generated by: MATCH(app_pwm_m_edge3)
  //state machine: do not change state
  //misc control: limiting
  //output control: updated from the isr based on the rotor position and direction
  SCT0->EV[6].STATE = 1<<app_pwm_st_active;
  SCT0->EV[6].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_edge3) | EVCTR_STATE_NO_CHANGE;
  SCT0->LIMIT |= 1<<6;
  
  //setup for event: app_pwm_ev_adc_trigger
  //enabled in state: app_pwm_st_active
  //generated by: MATCH(app_pwm_m_adc_trigger)
  //state machine: do not change state
  //output control: toggle app_out_adc_trigger
  SCT0->EV[7].STATE = 1<<app_pwm_st_active;
  SCT0->EV[7].CTRL = SCT_EV_CTRL_HEVENT(0) | EVCTR_MATCH_EVENT(app_pwm_m_adc_trigger) | EVCTR_STATE_NO_CHANGE;
  SCT0->OUT[app_out_adc_trigger].SET |= 1<<7;
  SCT0->OUT[app_out_adc_trigger].CLR |= 1<<7;
  
  //debug halt event is 8th event
  //debug halt may be generated when application is in idle or active or transient state
  SCT0->EV[8].STATE = (
                       (1<<app_pwm_st_idle) | 
                       (1<<app_pwm_st_active) | 
                       (1<<app_pwm_st_transient)
                      );
  SCT0->EV[8].CTRL = SCT_EV_CTRL_DIRECTION(0x0) | //This event is triggered 
                                                  //regardless of the count direction
                     SCT_EV_CTRL_IOCOND(0x3) |    //high level is trigger condition
                     SCT_EV_CTRL_OUTSEL(0x0) |    //trigger source is input
                     SCT_EV_CTRL_IOSEL(0x7) |     //debug halt is connected to SCTimer input 7
                     SCT_EV_CTRL_COMBMODE(0x2) |  //only IO event can trigger this event
                     EVCTR_STATE_NO_CHANGE;       //state no change
  SCT0->OUT[app_out_pwm1].CLR |= (1<<8);//only event 8 can clear PWM out1
  SCT0->OUT[app_out_pwm1].SET = 0;
  SCT0->OUT[app_out_pwm2].CLR |= (1<<8);//only event 8 can clear PWM out2
  SCT0->OUT[app_out_pwm2].SET = 0;
  SCT0->OUT[app_out_pwm3].CLR |= (1<<8);//only event 8 can clear PWM out3
  SCT0->OUT[app_out_pwm3].SET = 0;
  SCT0->OUT[app_out_pwm4].CLR |= (1<<8);//only event 8 can clear PWM out4
  SCT0->OUT[app_out_pwm4].SET = 0;
  SCT0->OUT[app_out_pwm5].CLR |= (1<<8);//only event 8 can clear PWM out5
  SCT0->OUT[app_out_pwm5].SET = 0;
  SCT0->OUT[app_out_pwm6].CLR |= (1<<8);//only event 8 can clear PWM out6
  SCT0->OUT[app_out_pwm6].SET = 0;
  SCT0->STOP |= (1<<8);//stop SCTimer when event 8 occurs
  /******************************SCTimer event setup*******************************************************/
  
  /******************************Make sure no PWM output activity is selected at this point****************/
//  SCT0->OUT[app_out_pwm1].CLR = 0;
//  SCT0->OUT[app_out_pwm1].SET = 0;
//  SCT0->OUT[app_out_pwm2].CLR = 0;
//  SCT0->OUT[app_out_pwm2].SET = 0;
//  SCT0->OUT[app_out_pwm3].CLR = 0;
//  SCT0->OUT[app_out_pwm3].SET = 0;
//  SCT0->OUT[app_out_pwm4].CLR = 0;
//  SCT0->OUT[app_out_pwm4].SET = 0;
//  SCT0->OUT[app_out_pwm5].CLR = 0;
//  SCT0->OUT[app_out_pwm5].SET = 0;
//  SCT0->OUT[app_out_pwm6].CLR = 0;
//  SCT0->OUT[app_out_pwm6].SET = 0;
  /******************************Make sure no PWM output activity is selected at this point****************/
  
  /******************************Set the initial state of the SCTimer state machine************************/
  SCT0->STATE = app_pwm_st_idle;
  /******************************Set the initial state of the SCTimer state machine************************/
  /******************************Set the initial value of SCTimer count************************************/
  SCT0->COUNT = 0xF00D;
  /******************************Set the initial value of SCTimer count************************************/
  /******************************Clear SCT event flag******************************************************/
  SCT0->EVFLAG = 0xFF;
  /******************************Clear SCT event flag******************************************************/
  
  /******************************Invalidate period estimate************************************************/
  invalidate_period_estimate();
  /******************************Invalidate period estimate************************************************/
  
  /******************************What to do when having SET/CLR at the same time***************************/
  SCT0->RES = 3<<(app_out_adc_trigger<<1);
  /******************************What to do when having SET/CLR at the same time***************************/

  /******************************SCTimer interrupt priority setting and interrupt enable******************/
  //Set interrupt priority
  NVIC_SetPriority(SCT0_IRQn, 1);
  //Enable SCTimer interrupt in NVIC 
  NVIC_EnableIRQ(SCT0_IRQn);
  /******************************SCTimer interrupt priority setting and interrupt enable******************/
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: MC_INIT
*
* @brief   The function is used to initiate PME and after call SCT init for motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void MC_INIT(void)
{
  //Initialize inputmux
  Inputmux_Init();
  //Initialize pattern match engine
  PME_Init();
  //Initialize SCTimer
  SCT_Init();
}	

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: SCT_STOP
*
* @brief   The function is used to stop SCT function.
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void SCT_STOP (void)
{
  //Halt both low part and high part of SCTimer
  SCT0->CTRL |=  SCT_CTRL_HALT_H(1) | SCT_CTRL_HALT_L(1);
  //All SCTimer outputs are low level
  SCT0->OUTPUT = 0x0;
  //invalidate_period_estimate();
}
/*********************************Function Definition************************************************/
