/*********************************Header File********************************************************/
#include "fsl_common.h"
#include "BLDC_app.h"
#include "BLDC_sct.h"
/*********************************Header File********************************************************/

/*********************************Type declaration***************************************************/
//Motor control state machine control function type
typedef void (*tPointerFcn)(void); 
/*********************************Type declaration***************************************************/

/*********************************Macro definition***************************************************/
#define APP_INIT                          0
#define APP_ALIGNMENT                     1
#define APP_START                         2
#define APP_RUN                           3
#define APP_STOP                          4
#define APP_ERROR                         5

#define I_DCB_MAX_AMP                   33.0 
#define SPEED_CTRL_I_LIMIT_HIGH         0.4
#define SPEED_CTRL_I_LIMIT_LOW          0.3
#define F16_SPEED_CTRL_I_LIMIT_HIGH     FRAC16(SPEED_CTRL_I_LIMIT_HIGH/I_DCB_MAX_AMP)
#define F16_SPEED_CTRL_I_LIMIT_LOW      FRAC16(SPEED_CTRL_I_LIMIT_LOW/I_DCB_MAX_AMP)
/*********************************Macro definition***************************************************/

/*********************************Function declaration***********************************************/
static void AppInit(void);
static void AppAlignment(void);
static void AppStart(void);
static void AppRun(void);
static void AppStop(void);
static void AppInitToStop(void);
static void AppStopToStart(void);
static void AppRunToStop(void);
/*********************************Function declaration***********************************************/

/*********************************Variable Definition************************************************/
uint8_t                    appState;
uint8_t                    ui8AppStartFlag;
frac16_t                   f16TargetSpeed;
frac16_t                   f16TargetSpeedRamped;
GFLIB_RAMP_T_F16           sSpeedRampParam;
frac16_t                   f16CaptureSpeed;
frac16_t                   f16SpeedCoef;
volatile uint32_t          speed_limit_monitor = 0;
volatile frac16_t          f16Idcb;
GFLIB_CTRL_PI_P_AW_T_A32   sSpeedCtrl;
volatile frac16_t          f16Vdcb;
frac16_t                   f16IdcbHighLimit;
frac16_t                   f16IdcbLowLimit;

int16_t                    cap_cnt;
int16_t                    cap_cntFilt;
frac16_t                   f16IdcbHighLimit;
frac16_t                   f16IdcbLowLimit;
GFLIB_CTRL_PI_P_AW_T_A32   sCurrentCtrl;
bool_t                     bStopIntegFlag;
volatile uint32_t          tick_limit = 45;
volatile uint32_t          dcb_cur,dcb_volt;
static   frac16_t          f16InErr;
static   frac16_t          f16PIout;
extern   unsigned char     g_InitPosVal;
extern   volatile uint16_t pwm_xy_deadtime;
extern   volatile uint32_t speed_timer_rd_new; 

static tPointerFcn AppStateMachine[] = 
{
  AppInit,
  AppAlignment,
  AppStart,
  AppRun,
  AppStop,
  //AppError
};
/*********************************Variable Definition************************************************/

/*********************************Function Definition************************************************/
/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: SysTick_Handler
*
* @brief   The function is used to control motor control state machine scheduling
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void SysTick_Handler(void)
{
  //monitoring variable for motor control program running increases 1
  speed_limit_monitor++;
  //motor control state machine scheduling
  AppStateMachine[appState]();
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: BLDC_AppStateMachineInit
*
* @brief  initialize app state machine
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void BLDC_AppStateMachineInit(void)
{
  //set state of application state machine as APP_INIT, this is initial state of application state machine
  appState = APP_INIT;
  //Set systick reload value to generate 100ms interrupt
  SysTick_Config(SystemCoreClock / 1000U);
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppInit
*
* @brief   motor control initialize
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppInit(void)
{
  /********************************1.Initialize BLDC application variables*****************************************************/
  f16IdcbLowLimit = F16_SPEED_CTRL_I_LIMIT_LOW;
  f16IdcbHighLimit = F16_SPEED_CTRL_I_LIMIT_HIGH;
  sSpeedCtrl.a32PGain = ACC32(0.5);
  sSpeedCtrl.a32IGain = ACC32(0.01);
  sSpeedCtrl.f16UpperLim = FRAC16(1.0);
  sSpeedCtrl.f16LowerLim = FRAC16(0.01);
  bStopIntegFlag = FALSE;
  GFLIB_CtrlPIpAWInit_F16(FRAC16(0.0), &sSpeedCtrl);

  sCurrentCtrl.a32PGain = ACC32(1);
  sCurrentCtrl.a32IGain = ACC32(0.01);
  sCurrentCtrl.f16UpperLim = FRAC16(1.0);
  sCurrentCtrl.f16LowerLim = FRAC16(0.0);
  GFLIB_CtrlPIpAWInit_F16(FRAC16(0.0), &sCurrentCtrl);
  
  sSpeedRampParam.f16RampUp   = FRAC16(0.01);
  sSpeedRampParam.f16RampDown = FRAC16(0.01);
  GFLIB_RampInit_F16(FRAC16(0.0), &sSpeedRampParam);
  f16SpeedCoef = FRAC16(0.07486);
  /********************************1.Initialize BLDC application variables*****************************************************/
  
  /********************************2.Intialize PME and SCTimer and Startup SCTimer-based PWM control state machine*************/
  cap_cnt = 0;
  MC_INIT();
  /********************************2.Intialize PME and SCTimer and Startup SCTimer-based PWM control state machine*************/
  
  /********************************3.Switch state of application state machine to Stop*****************************************/
  AppInitToStop();
  /********************************3.Switch state of application state machine to Stop*****************************************/
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppAlignment
*
* @brief   motor control alignment
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppAlignment(void)
{
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppAlignmentToStart
*
* @brief   Transit state Alignment -> Start in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppAlignmentToStart(void)
//{
//  //switch state machine from Alignment to Start
//  appState = APP_START;
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStartToRun
*
* @brief   Transit state Start -> Run in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppStartToRun(void)
{
  //switch state machine from Start to Run
  appState = APP_RUN;
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStart
*
* @brief   motor control startup
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppStart(void)
{
  //Unhalt SCT0 counter(Both part) to start up SCTimer
  SCT0->CTRL &= ~(SCT_CTRL_HALT_L_MASK | SCT_CTRL_HALT_H_MASK); 
  //control motor from start to run
  AppStartToRun();		
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStartToStop
*
* @brief   Transit state Start -> Stop in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppStartToStop(void)
//{
//  //switch state machine from Start to Stop
//  appState = APP_INIT;
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStartToError
*
* @brief   Transit state Start -> Error in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppStartToError(void)
//{
//  //switch state machine from Start to Error 
//  appState = APP_ERROR;
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppRun
*
* @brief   motor control running
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppRun(void)
{
  //PWM duty cycle
  static frac16_t duty;
  
  //Check whether motor is running or not
  if(speed_limit_monitor > tick_limit)
  {//No running
     f16CaptureSpeed=0;  
     speed_timer_rd_new = 0;
  }
  else
  {//running
    f16CaptureSpeed = MLIB_Div1QSat_F16(f16SpeedCoef, cap_cntFilt);  
  }
  
  //get target ramped speed
  f16TargetSpeedRamped = GFLIB_Ramp_F16(f16TargetSpeed, &sSpeedRampParam);
  
  if(f16Idcb > f16IdcbHighLimit)
  {//current on DCB is more than limit
    f16InErr = f16IdcbHighLimit - f16Idcb;
    f16PIout = GFLIB_CtrlPIpAW_F16(f16InErr, &bStopIntegFlag, &sCurrentCtrl);
    sSpeedCtrl.f32IAccK_1 = sCurrentCtrl.f32IAccK_1;
  }
  if(f16Idcb < f16IdcbLowLimit)
  {//current on DCB is less than limit
      f16InErr = f16TargetSpeedRamped - f16CaptureSpeed;
      f16PIout = GFLIB_CtrlPIpAW_F16(f16InErr, &bStopIntegFlag, &sSpeedCtrl);
      sCurrentCtrl.f32IAccK_1 = sSpeedCtrl.f32IAccK_1;
  }
  
  //get PWM duty cycle
  duty = MLIB_Mul_F16(CNT_MAXIMUN_VALUE, f16PIout);
  //update SCTimer match reload register 3 and 4
  MR3 = pwm_xy_deadtime+duty-1;
  MR4 = pwm_xy_deadtime+duty+pwm_xy_deadtime-1;
  
  //update ADC sampling point (@ 25% pulse x)
  MR6 = pwm_xy_deadtime+(duty>>2)-1;//>>1
  
  if (!ui8AppStartFlag)
  {//motor stop
    //stop SCTimer
    SCT_STOP();
    //clear SCTimer interrupt flag
    NVIC_ClearPendingIRQ((IRQn_Type) (SCT0_IRQn));
    //control motor from run to stop
    AppRunToStop();
  }
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppRunToError
*
* @brief   Transit state Run -> Error in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppRunToError(void)
//{
//  //switch state machine from Run to Error
//  appState = APP_ERROR;
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStop
*
* @brief   motor control stop
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppStop(void)
{
  if(ui8AppStartFlag)
  {
   //switch state machine from Stop to Start
   AppStopToStart();		
  }
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStopToError
*
* @brief   Transit state Stop -> Error in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppStopToError(void)
//{
//  //switch state machine from Stop to Error
//  appState = APP_ERROR;
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppError
*
* @brief   motor control error
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppError(void)
//{
//}


/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppBrake
*
* @brief   motor control brake
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppBrake(void)
//{
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppBrakeToStop
*
* @brief   Transit state Brake -> Stop in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
//static void AppBrakeToStop(void)
//{
//  //switch state machine Brake to Stop
//  appState = APP_STOP;
//}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppInitToStop
*
* @brief   Transit state Init -> Stop in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppInitToStop(void)
{
  //switch state machine from Init to Stop
  appState = APP_STOP; 
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppStopToStart
*
* @brief   Transit state Stop -> Start in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppStopToStart(void)
{
  //switch state machine from Stop to Start
  appState = APP_START;  
}

/*****************************************************************************
+FUNCTION----------------------------------------------------------------
* @function name: AppRunToStop
*
* @brief   Transit state Run -> Stop in motor control
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static void AppRunToStop(void)
{
  //switch state machine from Run to Stop
  appState = APP_INIT;
}
/*********************************Function Definition************************************************/