/* touch.c */

#include "touch.h"
#include "fsl_clock.h"
#include "fsl_port.h"
#include "fsl_tsi_v5.h"

/* TSI转换器通道号 */
const uint8_t touch_channel_code[TOUCH_2D_ALL_COUNT] =
{
    15, 16, 22, 23,  7,  6,   /* X. */
    17, 10, 18, 14, 13, 11,   /* Y. */
    12    /* Shield. */
};

volatile uint32_t touch_channel_index;
volatile int32_t  touch_channel_values[TOUCH_2D_ALL_COUNT]; /* 扩展了采样结果寄存器. */
volatile bool     touch_scan_queue_done = false;

/* 初始化引脚 */
void touch_init_pins(void)
{
    /* enable clocks. */
    CLOCK_EnableClock(kCLOCK_PortA);
    CLOCK_EnableClock(kCLOCK_PortC);
    CLOCK_EnableClock(kCLOCK_PortD);
    CLOCK_EnableClock(kCLOCK_PortE);

    /* y. */
    PORT_SetPinMux(PORTC, 6u , kPORT_PinDisabledOrAnalog); /* R0. */
    PORT_SetPinMux(PORTC, 7u , kPORT_PinDisabledOrAnalog); /* R1. */
    PORT_SetPinMux(PORTC, 0u , kPORT_PinDisabledOrAnalog); /* R2. */
    PORT_SetPinMux(PORTC, 1u , kPORT_PinDisabledOrAnalog); /* R3. */
    PORT_SetPinMux(PORTD, 6u , kPORT_PinDisabledOrAnalog); /* R4. */
    PORT_SetPinMux(PORTD, 5u , kPORT_PinDisabledOrAnalog); /* R5. */

    /* x. */
    PORT_SetPinMux(PORTA, 0u , kPORT_PinDisabledOrAnalog); /* C0. */
    PORT_SetPinMux(PORTD, 7u , kPORT_PinDisabledOrAnalog); /* C1. */
    PORT_SetPinMux(PORTA, 1u , kPORT_PinDisabledOrAnalog); /* C2. */
    PORT_SetPinMux(PORTE, 1u , kPORT_PinDisabledOrAnalog); /* C3. */
    PORT_SetPinMux(PORTE, 0u , kPORT_PinDisabledOrAnalog); /* C4. */
    PORT_SetPinMux(PORTE, 8u , kPORT_PinDisabledOrAnalog); /* C5. */

    /* shield. */
    PORT_SetPinMux(PORTC, 5u , kPORT_PinDisabledOrAnalog); /* shield. */
}

/* 使用中断方式在后台扫描传感器队列. */
void touch_init(void)
{
    TSI_Type *base = TSI;

    CLOCK_EnableClock(kCLOCK_Tsi0);

    base->GENCS = TSI_GENCS_TSIEN(1)  /* enable tsi converter. */
                | TSI_GENCS_DVOLT(0)  /* 采样比较器的门限阈值. */
                | TSI_GENCS_ESOR(0)   /* enable end-of-scan interrupt. */
                | TSI_GENCS_TSIIEN(1) /* enable interrupts. */
                | TSI_GENCS_STPE(0)   /* disable tsi in low power modes. */
                | TSI_GENCS_STM(0)    /* software trigger. */
                ;

    /* setup filter. */
    base->SINC  = TSI_SINC_CUTOFF(1)     /* 移除尾巴末尾n位. */
                | TSI_SINC_ORDER(0)      /* use 1 or 2 order SINC filter. */
                | TSI_SINC_DECIMATION(8) /* 为每个通道连续采样次数n倍 */
                ;

    /* clear flags and enable interrupt. */
    base->GENCS |= TSI_GENCS_TSIIEN(1) /* enable interrupts. */
                |  TSI_GENCS_ESOR(1)   /* enable end-of-scan interrupt. */
                ;

    NVIC_EnableIRQ(TSI_IRQn);
}

void touch_do_soft_trigger(uint32_t channel_index)
{
    TSI_Type *base = TSI;

    base->DATA  = TSI_DATA_SWTS_MASK
                | TSI_DATA_TSICH(channel_index)
                ;
}

uint32_t touch_get_conv_value(void)
{
    TSI_Type *base = TSI;
    return (TSI_DATA_TSICNT_MASK & base->DATA);
}

void touch_start(void)
{
    touch_scan_queue_done = false;
    touch_channel_index = 0u;
    touch_do_soft_trigger(touch_channel_code[touch_channel_index]);
}

/* ISR for TSI interrupt. */
void TSI_IRQHandler(void)
{
    TSI_Type *base = TSI;
    uint32_t flags = base->GENCS;
    base->GENCS = flags; /* clear flags */

    if (flags & TSI_GENCS_EOSF_MASK)
    {
        if (touch_channel_index < (TOUCH_2D_ALL_COUNT))
        {
            touch_channel_values[touch_channel_index] = touch_get_conv_value();
            touch_channel_index++;
            touch_do_soft_trigger(touch_channel_code[touch_channel_index]);
        }
        else
        {
            touch_scan_queue_done = true;
        }
    }
}

void touch_wait_data_ready(int32_t *outputs)
{
    while (!touch_scan_queue_done)
    {}
    touch_scan_queue_done = false;

    for (uint32_t i = 0u; i < TOUCH_2D_ALL_COUNT; i++)
    {
        outputs[i] = touch_channel_values[i];
    }
}

/* EOF. */

