/* main.c */

#include <stdint.h>
#include <stdbool.h>
#include "fsl_common.h"

#include "fsl_clock.h"
#include "fsl_gpio.h"
#include "fsl_port.h"
#include "fsl_llwu.h"

void disable_lvd_lvm(void);
void disable_clock_gates(void);
void disable_pins(void);
void wait_user_button(void);
void setup_wakeup_button(void);
void enter_vlls3(void);

/*
 * System will boot-up in BLPI (Bypass Low Power Internal) with 4 MHz IRC being used and FLL disabled.
 * All module clocks except RTC clock are disabled. iRTC is the only module that can function after POR as it is in separate power domain (VBAT).
 *  */
int main(void)
{
    /* disable the low voltage detection/warning. */
    disable_lvd_lvm();

    /* clear the io pin lock. */
    if (   (RCM->SRS0 == 0x41)   /* VLLS mode wake up due to RESET pin assertion. */
        || (RCM->SRS0 == 0x01) ) /* VLLS mode wake up due to other wake up sources. */
    {
        PMC->REGSC |= PMC_REGSC_ACKISO_MASK;
        NVIC_ClearPendingIRQ(LLWU_IRQn);
    }

    /* pending before enter the low power modes. */
    wait_user_button();

    /* disable the modules. */
    disable_pins();
    disable_clock_gates();

    /* setup wake up source. */
    setup_wakeup_button();

    /* enter the low power vlls3 mode. */
    enter_vlls3();

    /* the code should never run to here. */
    while (1);
}

/* lvd: low voltage detection.
 * lvm: low voltage warning.
 */
void disable_lvd_lvm(void)
{
    PMC->LVDSC1 = 0x00;
    PMC->LVDSC2 = 0x00;
}

/* SW2: PTD1. */
void wait_user_button(void)
{
    CLOCK_EnableClock(kCLOCK_PortD);

    PORTD->PCR[1] = PORT_PCR_MUX(1) | PORT_PCR_PE_MASK | PORT_PCR_PS(1); /* gpio pin with pull-up. */

    gpio_pin_config_t gpio_pin_config;
    gpio_pin_config.pinDirection = kGPIO_DigitalInput;
    GPIO_PinInit(GPIOD, 1u, &gpio_pin_config);

    while (1u == GPIO_PinRead(GPIOD, 1u))
    {}

    while (0u == GPIO_PinRead(GPIOD, 1u))
    {}

    PORTD->PCR[1] = PORT_PCR_MUX(0); /* gpio pin disabled. */
    CLOCK_DisableClock(kCLOCK_PortD);
}

/* SW1: PTA4/LLWU_P15. */
void setup_wakeup_button(void)
{
    CLOCK_EnableClock(kCLOCK_PortA);
    PORTA->PCR[4] = PORT_PCR_MUX(1); /* PTA4/LLWU_P15. */
    CLOCK_DisableClock(kCLOCK_PortA);

    /* setup for vlls3/2/1/0. */
    LLWU_SetExternalWakeupPinMode(LLWU, 15u, kLLWU_ExternalPinRisingEdge); /* LLWU_P15 */
    NVIC_EnableIRQ(LLWU_IRQn);
}

void enter_vlls3(void)
{
    /* unlock the modes. */
    SMC->PMPROT = SMC_PMPROT_AVLLS_MASK;

    /* prepare to enter the STOP modes. */
    SMC->PMCTRL = SMC_PMCTRL_RUNM(0x00) | SMC_PMCTRL_STOPM(0x04);
    SMC->STOPCTRL = SMC_STOPCTRL_PSTOPO(0x00)   // 2'b00 normal stop; 2'b01 pstop1; 2'b02 pstop2
                  | SMC_STOPCTRL_VLLSM(0x03); /* vlls3. */

    /* stop entry. */
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    asm
    (
        "wfi;"
        "nop;"
    );
}

void disable_clock_gates(void)
{
    SIM->SCGC4 = 0x0;
    SIM->SCGC5 = 0x0;
    SIM->SCGC6 = SIM_SCGC6_FTFA_MASK;
    SIM->SCGC7 = 0x0;
}

void disable_pins(void)
{
    CLOCK_EnableClock(kCLOCK_PortE);
    PORTE->PCR[6] = PORT_PCR_MUX(0); /* swdio. */
    PORTE->PCR[7] = PORT_PCR_MUX(0); /* swclk. */
    CLOCK_DisableClock(kCLOCK_PortE);
}

/*!
 * @brief LLWU interrupt handler.
 */
void LLWU_IRQHandler(void)
{
    /* if waken up by external pin. */
    if (LLWU_GetExternalWakeupPinFlag(LLWU, 15u))
    {
        /* clear the pin interrupt. */
        //PORT_SetPinInterruptConfig(PORTA, 4u, kPORT_InterruptOrDMADisabled);
        //PORT_ClearPinsInterruptFlags(PORTA, (1U << 4u));

        /* clear llwu flag. */
        LLWU_ClearExternalWakeupPinFlag(LLWU, 15u);
    }

    __DSB();
}

/*
void PTx_IRQHandler(void)
{
    if ((1U << 4u) & PORT_GetPinsInterruptFlags(PORTA))
    {
        PORT_SetPinInterruptConfig(PORTA, 4u, kPORT_InterruptOrDMADisabled);
        PORT_ClearPinsInterruptFlags(PORTA, (1U << 4u));
    }
    __DSB();
}
*/

/* EOF. */

