/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <UART_main.h>
#include "board.h"
#include "fsl_uart.h"
#include "fsl_clock.h"
#include "ApplMain.h"
#include "ble_FSCI.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "FSCI_main.h"
#include "timers.h"
#include "timer_manager.h"
#include "SPI.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

#ifdef gUARTSupported

/* UART instance and clock */
#define UART3_CLK_FREQ CLOCK_GetFreq(UART3_CLK_SRC) //UART3_CLK_SRC
#define ECHO_BUFFER_LENGTH 64

#define PIN16_IDX                       16u   /*!< Pin number for pin 16 in a port */
#define PIN17_IDX                       17u   /*!< Pin number for pin 17 in a port */
#define SOPT5_UART0TXSRC_UART_TX      	0x00u   /*!< UART 0 transmit data source select: UART0_TX pin */
#define FSCI_SYNC_BYTE 					0x02

#define RCV_PACKET_HEADER_SIZE			5		/* 1 sync byte ,1 opgroup, 1 byte opcode, 2 byte length */

#define UART_TIMER_INTERVAL 			40
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/* UART user callback */
void UART3_UserCallback(UART_Type *base, uart_handle_t *handle, status_t status, void *userData);
static void UartTimreCallback(TimerHandle_t xTimer);

/*******************************************************************************
 * Private memory declarations
 ******************************************************************************/
static TimerHandle_t 	UARTTimerhandle = NULL;
static uint8_t		UARTTimerId;
/*******************************************************************************
 * Variables
 ******************************************************************************/
uint8_t  		masterTxDataUART[64] = {0U};
uint8_t  		masterRxDataUART[32][64] = {0U};
uint8_t 		g_rx3Buffer[ECHO_BUFFER_LENGTH] = {0};
uint16_t 		len = 0, respLen[32] = {0U};
uart_handle_t 	g_uart3Handle;
uart_transfer_t receive3Xfer;

static uint8_t 	j = 0, cnt = 0, cnt1 = 0, sendReqCnt = 0;

volatile bool rx3BufferEmpty            = true;
volatile bool tx3BufferFull             = false;
volatile bool tx3OnGoing                = false;
volatile bool rx3OnGoing                = false;
volatile bool isSyncByteReceived	= false;
/*******************************************************************************
 * Code
 ******************************************************************************/
extern void UART3_DriverIRQHandler(void);

/* UART user callback */
void UART3_UserCallback(UART_Type *base, uart_handle_t *handle, status_t status, void *userData)
{
	int index = 0;

	if (kStatus_UART_TxIdle == status)
    {
        tx3BufferFull = false;
        tx3OnGoing    = false;
    }

    if (kStatus_UART_RxIdle == status)
    {
        rx3OnGoing     = false;

        if(isSyncByteReceived == false)
        {
//        	calculate length of received packet
        	len = ((uint16_t)g_rx3Buffer[4] << 8) | g_rx3Buffer[3];
			receive3Xfer.dataSize = (len + 1);
			receive3Xfer.data     = g_rx3Buffer;

			isSyncByteReceived = true;
			for(j = 0; j < RCV_PACKET_HEADER_SIZE; j++)
			{
				masterRxDataUART[cnt][j] = g_rx3Buffer[j];
			}
			UART_TransferReceiveNonBlocking(UART3, &g_uart3Handle, &receive3Xfer, NULL);
        }

        else if(isSyncByteReceived == true)
		{
			for(index = 0; index < (len +1) ; index++)
			{
				masterRxDataUART[cnt][j++] = g_rx3Buffer[index];
			}
			respLen[cnt] = j;
			cnt++;
			len = 0;

			receive3Xfer.data     = g_rx3Buffer;
		    receive3Xfer.dataSize = RCV_PACKET_HEADER_SIZE;//ECHO_BUFFER_LENGTH;
			rx3OnGoing = true;

			UART_TransferReceiveNonBlocking(UART3, &g_uart3Handle, &receive3Xfer, NULL);
			isSyncByteReceived = false;
		}
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : BOARD_UARTInitPins
 * Description   : Configures pin for UART.
 *
 *END**************************************************************************/

void BOARD_UARTInitPins(void)
{
  CLOCK_EnableClock(kCLOCK_PortC);                           /* Port C Clock Gate Control: Clock enabled */

	PORT_SetPinMux(PORTC, PIN16_IDX, kPORT_MuxAlt3);           /* PORTC16 (pin 62) is configured as UART3_RX */
	PORT_SetPinMux(PORTC, PIN17_IDX, kPORT_MuxAlt3);           /* PORTC17 (pin 63) is configured as UART3_TX */
	SIM->SOPT5 = ((SIM->SOPT5 &
		(~(SIM_SOPT5_UART0TXSRC_MASK)))                          /* Mask bits to zero which are setting */
		  | SIM_SOPT5_UART0TXSRC(SOPT5_UART0TXSRC_UART_TX)       /* UART 0 transmit data source select: UART0_TX pin */
		);
}

void UARTInit(void)
{
    uart_config_t config;

    UART_GetDefaultConfig(&config);
    config.baudRate_Bps = BOARD_DEBUG_UART_BAUDRATE;
    config.enableTx     = true;
    config.enableRx     = true;

    UART_Init(UART3, &config, UART3_CLK_FREQ);
    UART_TransferCreateHandle(UART3, &g_uart3Handle, UART3_UserCallback, NULL);

	receive3Xfer.data     = g_rx3Buffer;
    receive3Xfer.dataSize = RCV_PACKET_HEADER_SIZE;//ECHO_BUFFER_LENGTH;

	if((!rx3OnGoing) && (!isSyncByteReceived))
	{
		rx3OnGoing = true;

		UART_TransferReceiveNonBlocking(UART3, &g_uart3Handle, &receive3Xfer, NULL);
	}

	UARTTimerhandle = xTimerCreate("UARTTimer", pdMS_TO_TICKS(UART_TIMER_INTERVAL), pdTRUE, &UARTTimerId, UartTimreCallback);
	if(UARTTimerhandle != NULL)
	{
		// Start timer
		xTimerStart(UARTTimerhandle, 0);
	}
}

void SendDataUART(uint8_t *buffer_ptr, uint16_t buffer_size)
{
	uint8_t index;
	uint32_t TRANSFER_SIZE;
	uart_transfer_t data_xfer_UART;

	TRANSFER_SIZE = buffer_size;

	for(index = 0; index < TRANSFER_SIZE; index++)
	{
		masterTxDataUART[index] = buffer_ptr[index];
	}


	data_xfer_UART.data = masterTxDataUART;
	data_xfer_UART.dataSize = TRANSFER_SIZE;

	if ((!tx3OnGoing))
	{
		tx3OnGoing = true;
		UART_TransferSendNonBlocking(UART3, &g_uart3Handle, &data_xfer_UART);
	}
	/* Wait send finished */
	while (tx3OnGoing)
	{
	}

	UART_EnableRx(UART3, true);
	sendReqCnt++ ;
}

/*! *********************************************************************************
* \brief   This thread checks continuously whether there is any message
* 		   available in UART.
*
********************************************************************************** */
static void UartTimreCallback(TimerHandle_t xTimer)
{
	if(cnt > cnt1)
	{
		uint8_t size = respLen[cnt1];
		uint8_t *send_data = malloc(size);
		uint8_t data = 0;

		memset(send_data, '\0', size);

		for(data = 0; data < size; data++)
		{
			send_data[data] = masterRxDataUART[cnt1][data];
			masterRxDataUART[cnt1][data] = 0U;
		}

		respLen[cnt1] = 0;
		cnt1++;

		OSA_MsgQPut(&rx_msg_queue, send_data);
		/* Let FSCI receive thread handle previous response */
		vTaskDelay(pdMS_TO_TICKS(20));
		/* Signal BLE Abstraction thread for response  message into message queue */
		OSA_EventSet(&evt_handle, gEvtMsgFromFSCI_c);

		item_cnt += 1;


		free(send_data);
		send_data = NULL;
	}
	if(cnt == cnt1)
	{
		cnt = cnt1 = 0;
	}
}

#endif	//gUARTSupported
