/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*System includes.*/
#include "ApplMain.h"
#include "ble_FSCI.h"
#include "gatt_database.h"
#include "gatt_types.h"
#include "gattdb_interface.h"
#include "fsl_debug_console.h"

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
uint16_t *primServiceHandle = NULL;
uint16_t *secServiceHandle = NULL;
uint16_t *charHandle = NULL;
uint16_t *charDescHandle = NULL;
uint16_t *cccdHandle = NULL;
uint16_t *servHandle = NULL;
uint16_t *charValHandle = NULL;
uint16_t *cccdHandleCharVal = NULL;
uint16_t *descHandle = NULL;
uint8_t  *attributeVal = NULL;
uint16_t  *attributeVallen = NULL;

bleResult_t GattDbDynamic_Init(void)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;
	BaseType_t result;

	/* Allocate a buffer with enough space to store the packet */
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBInit;
	msgToFSCI->length = 0;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_ReleaseDatabase(void)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBReleaseDB;
	msgToFSCI->length = 0;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_AddPrimaryService
(
    uint16_t        desiredHandle,
    bleUuidType_t   serviceUuidType,
    bleUuid_t*      pServiceUuid,
	uint16_t*       pOutHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;
	uint8_t uuidLen;
	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	memset(&msgToFSCI->bleRequestType.addPrimaryService, '\0', sizeof(gattDBAddPrimaryServiceReq_t));

	if(serviceUuidType == gBleUuidType16_c)
	{
		uuidLen = sizeof(uint16_t);
		bleUuid_t uuid;
		uuid.uuid16 = *(uint16_t*)pServiceUuid;
		msgToFSCI->bleRequestType.addPrimaryService.pServiceUuid = &uuid;
	}
	else if(serviceUuidType == gBleUuidType32_c)
	{
		uuidLen = sizeof(uint32_t);
		bleUuid_t uuid;
		uuid.uuid32 = *(uint32_t*)pServiceUuid;
		msgToFSCI->bleRequestType.addPrimaryService.pServiceUuid = &uuid;
	}
	else if(serviceUuidType == gBleUuidType128_c)
	{
		uuidLen = 8 * sizeof(uint16_t);
		bleUuid_t uuid;

		memcpy(&uuid.uuid128, &pServiceUuid->uuid128, 16);
		msgToFSCI->bleRequestType.addPrimaryService.pServiceUuid = &uuid;
	}

	msgToFSCI->opCode = gattDBAddPrimaryService;
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->bleRequestType.addPrimaryService.desireHandle = desiredHandle;
	msgToFSCI->bleRequestType.addPrimaryService.serviceUuidType = serviceUuidType;
	msgToFSCI->length = sizeof(gattDBAddPrimaryServiceReq_t) - sizeof(bleUuid_t*) + uuidLen;

	primServiceHandle = pOutHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(200));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_AddSecondaryService
(
    uint16_t        desiredHandle,
    bleUuidType_t   serviceUuidType,
    bleUuid_t*      pServiceUuid,
	uint16_t*       pOutHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opCode = gattDBAddSecondaryService;
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->length = sizeof(gattDBAddSecondaryServiceReq_t);
	msgToFSCI->bleRequestType.addSecondaryService.desireHandle = desiredHandle;
	msgToFSCI->bleRequestType.addSecondaryService.serviceUuidType = serviceUuidType;
	msgToFSCI->bleRequestType.addSecondaryService.pServiceUuid = pServiceUuid;

	secServiceHandle = pOutHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_AddCharacteristicDeclarationAndValue
(
    bleUuidType_t                               characteristicUuidType,
    bleUuid_t*                                  pCharacteristicUuid,
    gattCharacteristicPropertiesBitFields_t     characteristicProperties,
    uint16_t                                    maxValueLength,
    uint16_t                                    initialValueLength,
    uint8_t*                                    aInitialValue,
    gattAttributePermissionsBitFields_t         valueAccessPermissions,
	uint16_t*       							pOutHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;
	uint8_t uuidLen;
	osa_status_t result;

	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opCode = gattDBAddCharDescAndValue;
	msgToFSCI->opGroup = bleGATTDB;

	if(characteristicUuidType == gBleUuidType16_c)
	{
		uuidLen = sizeof(uint16_t);
		bleUuid_t uuid;
		uuid.uuid16 = *(uint16_t*)pCharacteristicUuid;
		msgToFSCI->bleRequestType.addCharDecandValue.pCharacteristicUuid = &uuid;
	}
	else if(characteristicUuidType == gBleUuidType32_c)
	{
		uuidLen = sizeof(uint32_t);
		bleUuid_t uuid;
		uuid.uuid32 = *(uint32_t*)pCharacteristicUuid;
		msgToFSCI->bleRequestType.addCharDecandValue.pCharacteristicUuid = &uuid;
	}
	else if(characteristicUuidType == gBleUuidType128_c)
	{
		uuidLen = 8 * sizeof(uint16_t);
		bleUuid_t uuid;

		memcpy(&uuid.uuid128, &pCharacteristicUuid->uuid128, 16);
		msgToFSCI->bleRequestType.addCharDecandValue.pCharacteristicUuid = &uuid;
	}

	/* Need to calculate payload length by adding individual's length */
	uint8_t size = sizeof(gattDBAddCharDecandValueReq_t) - sizeof(bleUuid_t*) - sizeof(uint8_t*) + initialValueLength + uuidLen;
	uint8_t initVal[initialValueLength];

	msgToFSCI->length = size;
	msgToFSCI->bleRequestType.addCharDecandValue.characteristicUuidType = characteristicUuidType;
	msgToFSCI->bleRequestType.addCharDecandValue.characteristicProperties = characteristicProperties;
	msgToFSCI->bleRequestType.addCharDecandValue.maxValueLength = maxValueLength;
	msgToFSCI->bleRequestType.addCharDecandValue.initialValueLength = initialValueLength;
	msgToFSCI->bleRequestType.addCharDecandValue.aInitialValue = initVal;
	memcpy(msgToFSCI->bleRequestType.addCharDecandValue.aInitialValue, aInitialValue, initialValueLength);
	msgToFSCI->bleRequestType.addCharDecandValue.valueAccessPermissions = valueAccessPermissions;

	charHandle = pOutHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(200));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_AddCharacteristicDescriptor
(
    bleUuidType_t                               descriptorUuidType,
    bleUuid_t*                                  pDescriptorUuid,
    uint16_t                                    descriptorValueLength,
    uint8_t*                                    aInitialValue,
    gattAttributePermissionsBitFields_t         descriptorAccessPermissions,
	uint16_t*       							pOutHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;
	uint8_t uuidLen;
	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opCode = gattDBAddCharDescriptor;
	msgToFSCI->opGroup = bleGATTDB;

	if(descriptorUuidType == gBleUuidType16_c)
	{
		uuidLen = sizeof(uint16_t);
		bleUuid_t uuid;
		uuid.uuid16 = *(uint16_t*)pDescriptorUuid;
		msgToFSCI->bleRequestType.addCharDescriptor.pDescriptorUuid = &uuid;
	}
	else if(descriptorUuidType == gBleUuidType32_c)
	{
		uuidLen = sizeof(uint32_t);
		bleUuid_t uuid;
		uuid.uuid32 = *(uint32_t*)pDescriptorUuid;
		msgToFSCI->bleRequestType.addCharDescriptor.pDescriptorUuid = &uuid;
	}
	else if(descriptorUuidType == gBleUuidType128_c)
	{
		uuidLen = 8 * sizeof(uint16_t);
		bleUuid_t uuid;

		memcpy(&uuid.uuid128, &pDescriptorUuid->uuid128, 16);
		msgToFSCI->bleRequestType.addCharDescriptor.pDescriptorUuid = &uuid;
	}

	/* Need to calculate payload length by adding individual's length */
	uint8_t size = sizeof(gattDBAddCharacDescriptorReq_t) - sizeof(bleUuid_t*) - sizeof(uint8_t*) + descriptorValueLength + uuidLen;
	uint8_t initVal[descriptorValueLength];

	msgToFSCI->length = size;
	msgToFSCI->bleRequestType.addCharDescriptor.descriptorUuidType = descriptorUuidType;
	msgToFSCI->bleRequestType.addCharDescriptor.pDescriptorUuid = pDescriptorUuid;
	msgToFSCI->bleRequestType.addCharDescriptor.descriptorValueLength = descriptorValueLength;
	msgToFSCI->bleRequestType.addCharDescriptor.aInitialValue = initVal;
	memcpy(msgToFSCI->bleRequestType.addCharDescriptor.aInitialValue, aInitialValue, descriptorValueLength);
	msgToFSCI->bleRequestType.addCharDescriptor.descriptorAccessPermission = descriptorAccessPermissions;

	charDescHandle = pOutHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(200));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_AddCccd
(
		uint16_t*       pOutHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBAddCccd;
	msgToFSCI->length = 0;

	cccdHandle = pOutHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(200));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_WriteAttribute
(
    uint16_t    handle,
    uint16_t    valueLength,
    uint8_t*    aValue
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBWriteAttribute;
	msgToFSCI->length = sizeof(gattDBWriteAttributeReq_t) - sizeof(uint8_t*) + valueLength;
	msgToFSCI->bleRequestType.writeAttribute.handle = handle;
	msgToFSCI->bleRequestType.writeAttribute.valueLength = valueLength;
	msgToFSCI->bleRequestType.writeAttribute.aValue = aValue;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(200));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_ReadAttribute
(
    uint16_t    handle,
    uint16_t    maxBytes,
	uint8_t*    aOutValue,
	uint16_t*   pOutValueLength
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBReadAttribute;
	msgToFSCI->length = sizeof(gattDBReadAttributeReq_t);
	msgToFSCI->bleRequestType.readAttribute.handle = handle;
	msgToFSCI->bleRequestType.readAttribute.maxBytes = maxBytes;

	attributeVal = aOutValue;
	attributeVallen = pOutValueLength;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_FindServiceHandle
(
    uint16_t        startHandle,
    bleUuidType_t   serviceUuidType,
    bleUuid_t*      pServiceUuid,
	uint16_t*       pOutServiceHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBFindServiceHandle;
	msgToFSCI->length = sizeof(gattDBFindSeriveHandleReq_t);
	msgToFSCI->bleRequestType.findServiceHandle.startHandle = startHandle;
	msgToFSCI->bleRequestType.findServiceHandle.serviceUuidType = serviceUuidType;
	msgToFSCI->bleRequestType.findServiceHandle.pServiceUuid = pServiceUuid;

	servHandle = pOutServiceHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_FindCharValueHandleInService
(
    uint16_t        serviceHandle,
    bleUuidType_t   characteristicUuidType,
    bleUuid_t*      pCharacteristicUuid,
	uint16_t*       pOutCharValueHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;
	uint8_t uuidLen;
	osa_status_t result;

	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBFindCharValueHandleInService;

	if(characteristicUuidType == gBleUuidType16_c)
	{
		uuidLen = sizeof(uint16_t);
		bleUuid_t uuid;
		uuid.uuid16 = *(uint16_t*)pCharacteristicUuid;
		msgToFSCI->bleRequestType.findCharValueHandleInService.pCharacteristicUuid = &uuid;
	}
	else if(characteristicUuidType == gBleUuidType32_c)
	{
		uuidLen = sizeof(uint32_t);
		bleUuid_t uuid;
		uuid.uuid32 = *(uint32_t*)pCharacteristicUuid;
		msgToFSCI->bleRequestType.findCharValueHandleInService.pCharacteristicUuid = &uuid;
	}
	else if(characteristicUuidType == gBleUuidType128_c)
	{
		uuidLen = 8 * sizeof(uint16_t);
		bleUuid_t uuid;

		memcpy(&uuid.uuid128, &pCharacteristicUuid->uuid128, 16);
		msgToFSCI->bleRequestType.findCharValueHandleInService.pCharacteristicUuid = &uuid;
	}
	msgToFSCI->length = sizeof(gattDBFindCharValueHandleInServiceReq_t) - sizeof(bleUuid_t*) + uuidLen;
	msgToFSCI->bleRequestType.findCharValueHandleInService.serviceHandle = serviceHandle;
	msgToFSCI->bleRequestType.findCharValueHandleInService.characteristicUuidType = characteristicUuidType;

	charValHandle = pOutCharValueHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);
	vTaskDelay(pdMS_TO_TICKS(200));
	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_FindCccdHandleForCharValueHandle
(
    uint16_t        charValueHandle,
	uint16_t*       pOutCccdHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBFindCccdHandleForCharValHandle;
	msgToFSCI->length = sizeof(gattDBFindCccdHandleForCharValueHandleReq_t);
	msgToFSCI->bleRequestType.findCccdHandleForCharValueHandle.charValueHandle = charValueHandle;

	cccdHandleCharVal = pOutCccdHandle;

	OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(200));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}

bleResult_t GattDb_FindDescriptorHandleForCharValueHandle
(
    uint16_t        charValueHandle,
    bleUuidType_t   descriptorUuidType,
    bleUuid_t*      pDescriptorUuid,
	uint16_t*       pOutDescriptorHandle
)
{
	msgBLEabsToFSCI_t *msgToFSCI = NULL;
	uint32_t ulNotifiedValue;

	osa_status_t result;
	msgToFSCI = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));
	msgToFSCI->opGroup = bleGATTDB;
	msgToFSCI->opCode = gattDBFindDescHandleForCharvalHandle;
	msgToFSCI->length = sizeof(gattDBFindDescHandleForCharValueHandleReq_t);
	msgToFSCI->bleRequestType.findDescHandleForCharValueHandle.charValueHandle = charValueHandle;
	msgToFSCI->bleRequestType.findDescHandleForCharValueHandle.descriptorUuidType = descriptorUuidType;
	msgToFSCI->bleRequestType.findDescHandleForCharValueHandle.pDescriptorUuid = pDescriptorUuid;

	descHandle = pOutDescriptorHandle;

	result = OSA_MsgQPut(&tx_msg_queue, msgToFSCI);

	/* Signal FSCI thread for message into message queue */
	OSA_EventSet(&eventHandle, gFSCIEvtMsgFromAbs_c);

	result = xTaskNotifyWait(0x00, 0, &ulNotifiedValue, pdMS_TO_TICKS(1000));

	free(msgToFSCI);
	msgToFSCI = NULL;

	if(result == pdTRUE)
	{
		return ulNotifiedValue;
	}
	else
	{
		return RESP_TIMEOUT_ERROR;
	}
}
