/*! *********************************************************************************
* \addtogroup RSSI ranging application
* @{
********************************************************************************** */
/*! *********************************************************************************
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* Copyright 2016-2018 NXP
* All rights reserved.
*
* \file
*
* SPDX-License-Identifier: BSD-3-Clause
********************************************************************************** */

/************************************************************************************
 *************************************************************************************
 * Include
 *************************************************************************************
 ************************************************************************************/
#include <math.h>
#include "stdio.h"
#include "stdint.h"
#include "stdbool.h"
#include "RSSI_ranging.h"
#include "FunctionLib.h"
/************************************************************************************
 *************************************************************************************
 * Private macros
 *************************************************************************************
 ************************************************************************************/


/************************************************************************************
 *************************************************************************************
 * Private memory declarations
 *************************************************************************************
 ************************************************************************************/
static float     mAttenuationFactor = 3.0;
static int8_t    mRssiValueAtOneMeter = -52;
int8_t    mArrayStoreRssi[gRSSIDefaultSampls];
uint32_t  mRssiSampleCount ;

/************************************************************************************
*************************************************************************************
* Private functions declarations
*************************************************************************************
************************************************************************************/
static float RSSI_CalcStandDeviation 
(
    int8_t * pValue, 
    uint32_t n, 
    float average, 
    StandardDeviationType_t type
);

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/*! *********************************************************************************
* \brief        The distance is calculated by RSSI
*
*  RSSI = A - ( 10 * n * lg D )
*
*  RSSI:Rx RSSI value when ranging
*  A: Rx RSSI value at a 1 m
*  n: attenuation factor
*  D: distance
*            ||
*            ||
*           \  /
*           \ /        
*  D = 10 ^ [( A - RSSI ) / (10 * n)]
*
* \param[in]    rssi        RSSI that measured value.
********************************************************************************** */
float RSSI_DistanceCalc(float rssi)
{
    //get RSSI value
    float mRssi = rssi;
    
    //calculate power
    float power = ( (float)mRssiValueAtOneMeter - mRssi ) / ( 10.0 * mAttenuationFactor );

    return pow(10, power);
}

/*! *********************************************************************************
* \brief        Set the gRssiValueAtOneMeter value
*
* \param[in]    refer_value      RSSI value at a 1 m.
********************************************************************************** */
void RSSI_SetRssiReferenceValue(int8_t refer_value)
{
    mRssiValueAtOneMeter = refer_value;
}

/*! *********************************************************************************
* \brief        Get the gRssiValueAtOneMeter value
*
* \param[out]   gRssiValueAtOneMeter
********************************************************************************** */
int8_t RSSI_GetRssiReferenceValue(void)
{
     return mRssiValueAtOneMeter;
}

/*! *********************************************************************************
* \brief        Set the gAttenuationFactor value
*
* \param[in]    factory        attenuation factor.
********************************************************************************** */
void RSSI_SetAttenuationFactor(float factory)
{
    /* If the attenuation factory is 0, it means that the signal is not attenuated 
       in the air, then we cannot calculate the distance using RSSI. */
    if( factory != 0 )
    {
        mAttenuationFactor = factory;
    }
}

/*! *********************************************************************************
* \brief        Get the gAttenuationFactor value
*
* \param[out]   gAttenuationFactor.
********************************************************************************** */
float RSSI_GetAttenuationFactor(void)
{
     return mAttenuationFactor;
}



/************************************************************************************
* Data filtering algorithm
************************************************************************************/

/*! *********************************************************************************
* \brief       Calculate the average of number data 
*
* \param[in]   pValue             Pointer to input RSSI data     
* \param[in]   number             number of data that need to calculate
*
* \return                         average value.
********************************************************************************** */
float RSSI_CalcAverage (int8_t * pValue , uint32_t number ) //average filtering
{
    float avg = 0.0;
    uint32_t i;
    
    //get sum
    for (i = 0; i < number; i++)
    {
        avg += pValue[i];  
    }
    
    //calculate the average
    avg = avg / (float)number;
    
    return avg;
}

/*! *********************************************************************************
* \brief       Sort number data by value, and select the middle value
*      
* \param[in]   pValue             Pointer to input RSSI data     
* \param[in]   number             number of data 
*
* \return                         middle value.
********************************************************************************** */
float RSSI_MiddleSmoothing (int8_t * pValue , uint32_t number )
{
    uint32_t i, j, k;
    int8_t temp;
    
    /* sort */
    for ( i = 0; i < number - 1; i++) 
    {
        k = i;
        for (j = i + 1; j < number; j++)
        {
            if ( pValue[j] < pValue[k])
            {
                k = j;
            }
        }
        temp = pValue[k];
        pValue[k] = pValue[i];
        pValue[i] = temp;
    }
    
    /* select the middle value*/
    if(( number % 2 ) == 0)
    {   /* if number is even then take the middle two values and average them */ 
        return ((float)( pValue[ number / 2 ] + pValue[( number / 2 ) - 1] ) / 2.0 );
    }
    else
    {
        return  (float)( pValue[ number / 2 ] );
    }
}

/*! *********************************************************************************
* \brief       Calculate the standard deviation of n data 
*
* \param[in]   pValue             Pointer to  input RSSI data     
* \param[in]   n                  number of data 
* \param[in]   type               type of standard deviation
*                                                 
* \param[out]                     standard deviation.
********************************************************************************** */
static float RSSI_CalcStandDeviation 
(
    int8_t * pValue, 
    uint32_t n, 
    float average, 
    StandardDeviationType_t type
)
{
    float dev,sum = 0.0;   
    uint32_t i; 
      
    if(( n == 0) || ( n == 1 ))
    {
        /* standard deviation is 0 when the number of data less than 1 */
        return 0.0;
    }
    
    for (i = 0; i < n; i++)
    {
        sum += pow((pValue[i] - average), 2);  
    }
        
    if( type == SD_sample )
    { 
        dev = sum/(float)(n-1);
    }
    else
    {
        dev = sum/(float)n;
    }

    return sqrt(dev);
}

/*! *********************************************************************************
* \brief       Gaussian Smoothing
* \description Select the RSSI data which rssi value is within the gaussian interval. 
*
* \param[in]   pInValue           Pointer to input RSSI data     
* \param[in]   n                  number of data input
* \param[out]  pOutValue          Pointer to output RSSI data the gaussian interval
* \param[out]  pCount             data count in the gaussian interval                
*                                                 
* \return                         True if get data in gaussian interval, false otherwise. 
********************************************************************************** */
bool RSSI_GaussianSmoothing
(
    int8_t *   pInValue ,
    uint32_t   n ,
    int8_t *   pOutValue ,
    uint32_t * pCount 
)
{
    uint32_t i, num = 0;
    float avg  = 0.0;
    float StandDeviation = 0.0;   
    
    if(( n == 0) || ( n == 1 ))
    {
        /* standard deviation is 0 when the number of data less than 1 */
        return false;
    }
    
    /* calculate average */
    avg = RSSI_CalcAverage(pInValue, n);  
    /* calculate Stand Deviation */
    StandDeviation = RSSI_CalcStandDeviation ( pInValue, n, avg, SD_sample );
    
    /*In the case of low StandDeviationit may difficult to pick data which in Gaussian interval.
      For example: avg = 36.8 and StandDeviation = 0.1, gaussian interval is between 36.7-36.9
      this interval too small for RSSI integer to select data.*/
    if((StandDeviation * 2) < 1.0)
    { 
        *pCount = n;
        FLib_MemCpy(pOutValue, pInValue, n);
        return true;
        
    }
    
    for ( i = 0; i < n; i++)
    {
        if((pInValue[i] <= ( avg + StandDeviation )) && (pInValue[i] >= ( avg - StandDeviation )))
        {
            pOutValue[num] = pInValue[i];
            num += 1;
        }
    }
    
    if ( num == 0 || num > n )
    {
        return false;
    }
    
    *pCount = num;   

    return true;
}


/*! *********************************************************************************
* \brief       Gaussian mixed other filtering
* \description Select the RSSI data which rssi value is within the gaussian interval. 
*              Then calculate the avg or mid value or the avg of the first two.
*
* \param[in]   pValue             Pointer to input RSSI data     
* \param[in]   number             number of data that need to calculate
* \param[in]   MixType
*
* \return                         RSSI value.
********************************************************************************** */
float RSSI_GaussianMixedSmoothing (int8_t * pValue , uint32_t number , GaussianMixedSmoothingType_t MixType)
{
    int8_t StoreData[gRSSIDefaultSampls];
    uint32_t Count = 0; 
    float CalculatedRSSIavg = 0.0;
    float CalculatedRSSImid = 0.0;
    

    if(true == RSSI_GaussianSmoothing(pValue, number, StoreData, &Count))
    {   /* we get the data in gaussian interval, then we can calcuate*/
      
        /* average */
        if(MixType  == GS_CalcAvg)
        {
            CalculatedRSSIavg = RSSI_CalcAverage(StoreData, Count);
            return CalculatedRSSIavg;
        }       
        /* middle value */
        else if(MixType == GS_SelMidValue)
        {
            CalculatedRSSImid = RSSI_MiddleSmoothing(StoreData, Count);
            return CalculatedRSSImid;
        }
         /* middle + avg value */
        else if(MixType == GS_MixedAvgMid)
        {
             CalculatedRSSIavg = RSSI_CalcAverage(StoreData, Count);
             CalculatedRSSImid = RSSI_MiddleSmoothing(StoreData, Count);
             return (CalculatedRSSIavg + CalculatedRSSImid) / 2;
        }    
    }
    
    /* just return the all data avg if not get the data in gaussian interval*/
    CalculatedRSSIavg = RSSI_CalcAverage(pValue, number);
    return CalculatedRSSIavg;
}

