/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "fsl_lpspi.h"
#include "fsl_flexio_spi.h"
#include "board.h"

#include "fsl_common.h"
#include "pin_mux.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*Master related*/
#define TRANSFER_SIZE 1U        /*! Transfer dataSize */
#define TRANSFER_BAUDRATE 25000U /*! Transfer baudrate - 500k */


/*Slave related*/
#define SLAVE_FLEXIO_SPI_BASEADDR (FLEXIO1)
#define FLEXIO_SPI_SOUT_PIN 21U
#define FLEXIO_SPI_SIN_PIN 22U
#define FLEXIO_SPI_CLK_PIN 26U
#define FLEXIO_SPI_PCS_PIN 0U

#define SLAVE_FLEXIO_SPI_IRQ FLEXIO1_IRQn
/* Select USB1 PLL (480 MHz) as flexio clock source */
#define SLAVE_FLEXIO_SPI_CLOCK_SELECT (3U)
/* Clock divider for flexio clock source */
#define SLAVE_FLEXIO_SPI_CLOCK_PRE_DIVIDER (4U)
/* Clock divider for flexio clock source */
#define SLAVE_FLEXIO_SPI_CLOCK_DIVIDER (7U)


/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void FLEXIO_SPI_SlaveUserCallback(FLEXIO_SPI_Type *base,
                                  flexio_spi_slave_handle_t *handle,
                                  status_t status,
                                  void *userData);

/*******************************************************************************
 * Variables
 ******************************************************************************/
uint8_t slaveRxData[256]  = {0U};
uint8_t slaveTxData[1]  = {0U};
uint32_t j = 0;

FLEXIO_SPI_Type spiDev;
flexio_spi_slave_handle_t g_s_handle;

volatile bool isSlaveTransferCompleted  = false;

/*******************************************************************************
 * Code
 ******************************************************************************/

void FLEXIO_SPI_SlaveUserCallback(FLEXIO_SPI_Type *base,
                                  flexio_spi_slave_handle_t *handle,
                                  status_t status,
                                  void *userData)
{
    if (status == kStatus_Success)
    {
        __NOP();
    }

    isSlaveTransferCompleted = true;

    PRINTF("Slave send data: 0x%X\r\n", slaveTxData[j]);
    PRINTF("Slave receive data: 0x%X\r\n\r\n", slaveRxData[j]);
}

int main(void)
{
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    /* Clock setting for Flexio */
    CLOCK_SetMux(kCLOCK_Flexio1Mux, SLAVE_FLEXIO_SPI_CLOCK_SELECT);
    CLOCK_SetDiv(kCLOCK_Flexio1PreDiv, SLAVE_FLEXIO_SPI_CLOCK_PRE_DIVIDER);
    CLOCK_SetDiv(kCLOCK_Flexio1Div, SLAVE_FLEXIO_SPI_CLOCK_DIVIDER);

    PRINTF("FLEXIO SPI Slave example start.\r\n");
    PRINTF("Please make sure you make the correct line connection. Basically, the connection is:\r\n");
    PRINTF("FLEXIO_SPI_master -- FLEXIO_SPI_SLAVE   \r\n");
    PRINTF("       CLK        --       CLK  \r\n");
    PRINTF("       PCS        --       PCS  \r\n");
    PRINTF("       SOUT       --       SIN  \r\n");
    PRINTF("       SIN        --       SOUT \r\n");
    
    uint32_t i;

    flexio_spi_slave_config_t slaveConfig;
    flexio_spi_transfer_t slaveXfer;

    /* Slave config */
    FLEXIO_SPI_SlaveGetDefaultConfig(&slaveConfig);

    spiDev.flexioBase      = SLAVE_FLEXIO_SPI_BASEADDR;
    spiDev.SDOPinIndex     = FLEXIO_SPI_SOUT_PIN;
    spiDev.SDIPinIndex     = FLEXIO_SPI_SIN_PIN;
    spiDev.SCKPinIndex     = FLEXIO_SPI_CLK_PIN;
    spiDev.CSnPinIndex     = FLEXIO_SPI_PCS_PIN;
    spiDev.shifterIndex[0] = 0U;
    spiDev.shifterIndex[1] = 1U;
    spiDev.timerIndex[0]   = 0U;
    FLEXIO_SPI_SlaveInit(&spiDev, &slaveConfig);

    NVIC_SetPriority(SLAVE_FLEXIO_SPI_IRQ, 0U);

    slaveTxData[0] = 0xa5; 
    
    for(i=1; i<256; i++)
    {
        slaveTxData[i] = ~(i % 256);
    }
      
    isSlaveTransferCompleted = false;

    /* Set up slave first */
    FLEXIO_SPI_SlaveTransferCreateHandle(&spiDev, &g_s_handle, FLEXIO_SPI_SlaveUserCallback, NULL);

    /*Set slave transfer ready to receive/send data*/
    slaveXfer.txData   = slaveTxData;
    slaveXfer.rxData   = slaveRxData;
    slaveXfer.dataSize = TRANSFER_SIZE;
    slaveXfer.flags    = kFLEXIO_SPI_8bitMsb;

    FLEXIO_SPI_SlaveTransferNonBlocking(&spiDev, &g_s_handle, &slaveXfer);


    while (1)
    {
        i = 10000000;
        while(i)
        {
            i--;
        }
        
        j++;
        if(256 == j)
        {
            j = 0;
        }
        
        slaveXfer.txData   = &slaveTxData[j];
        FLEXIO_SPI_SlaveTransferNonBlocking(&spiDev, &g_s_handle, &slaveXfer);
    }
}
