/*
 * Copyright 2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "board.h"
#include "main.h"
#include "timer.h"
/*******************************************************************************
 * Variables
 ******************************************************************************/
int counter = 0;
const int no_of_samples = 2005;
/*******************************************************************************
 * Code
 ******************************************************************************/

static void i2c_release_bus_delay(void)
{
    uint32_t i = 0;
    for (i = 0; i < I2C_RELEASE_BUS_COUNT; i++)
    {
        __NOP();
    }
}

void BOARD_I2C_ReleaseBus(void)
{
    uint8_t i = 0;
    const gpio_pin_config_t pin_config = {.direction = kGPIO_DigitalOutput, .outputLogic = 1};

    CLOCK_EnableClock(kCLOCK_Iomuxc);

    IOMUXC_SetPinMux(IOMUXC_GPIO_AD_B1_01_GPIO1_IO17, 0U);
    IOMUXC_SetPinMux(IOMUXC_GPIO_AD_B1_00_GPIO1_IO16, 0U);

    GPIO_PinInit(ACCEL_I2C_SCL_GPIO, ACCEL_I2C_SCL_PIN, &pin_config);
    GPIO_PinInit(ACCEL_I2C_SDA_GPIO, ACCEL_I2C_SDA_PIN, &pin_config);

    /* Drive SDA low first to simulate a start */
    GPIO_PinWrite(ACCEL_I2C_SDA_GPIO, ACCEL_I2C_SDA_PIN, 0U);
    i2c_release_bus_delay();

    /* Send 9 pulses on SCL and keep SDA high */
    for (i = 0; i < 9; i++)
    {
        GPIO_PinWrite(ACCEL_I2C_SCL_GPIO, ACCEL_I2C_SCL_PIN, 0U);
        i2c_release_bus_delay();

        GPIO_PinWrite(ACCEL_I2C_SDA_GPIO, ACCEL_I2C_SDA_PIN, 1U);
        i2c_release_bus_delay();

        GPIO_PinWrite(ACCEL_I2C_SCL_GPIO, ACCEL_I2C_SCL_PIN, 1U);
        i2c_release_bus_delay();
        i2c_release_bus_delay();
    }

    /* Send stop */
    GPIO_PinWrite(ACCEL_I2C_SCL_GPIO, ACCEL_I2C_SCL_PIN, 0U);
    i2c_release_bus_delay();

    GPIO_PinWrite(ACCEL_I2C_SDA_GPIO, ACCEL_I2C_SDA_PIN, 0U);
    i2c_release_bus_delay();

    GPIO_PinWrite(ACCEL_I2C_SCL_GPIO, ACCEL_I2C_SCL_PIN, 1U);
    i2c_release_bus_delay();

    GPIO_PinWrite(ACCEL_I2C_SDA_GPIO, ACCEL_I2C_SDA_PIN, 1U);
    i2c_release_bus_delay();
}

int ACCEL_GetAccelData(int16_t *xp, int16_t *yp, int16_t *zp) {
	int16_t xData, yData, zData;
	float tmpx,tmpy,tmpz;
	fxos_data_t fxos_data = {0};
	/* Get new accelerometer data. */
	 if (FXOS_ReadSensorData(&g_fxosHandle, &fxos_data) != kStatus_Success)
	{
		PRINTF("Failed to read acceleration data!\r\n");
		return -1;
	}
    /* Get the X and Y data from the sensor data structure in 14 bit left format data*/
	xData = (int16_t)((uint16_t)((uint16_t)fxos_data.accelXMSB << 8) | (uint16_t)fxos_data.accelXLSB)/4U;
	yData = (int16_t)((uint16_t)((uint16_t)fxos_data.accelYMSB << 8) | (uint16_t)fxos_data.accelYLSB)/4U;
	zData = (int16_t)((uint16_t)((uint16_t)fxos_data.accelZMSB << 8) | (uint16_t)fxos_data.accelZLSB)/4U;

	tmpx = ((float)(xData)*g_dataScale*0.244);
	tmpy = ((float)(yData)*g_dataScale*0.244);
	tmpz = ((float)(zData)*g_dataScale*0.244);

    *xp = (int)(tmpx);
    *yp = (int)(tmpy);
    *zp = (int)(tmpz);

    return 0;
}

/*******************************************************************************
 * Code
 ******************************************************************************/
void GPT2_IRQHandler ()
{
	GPT_ClearStatusFlags(EXAMPLE_GPT, kGPT_OutputCompare1Flag);
	gptIsrFlag = true;
}

/* FUNCTION ************************************************************************************************************
 *
 * Function Name : Init_Accelerometere
 * Description   : Initialize the accelerometer from FXOS800CQ and configure a GPT for 1ms
 *
 * END ****************************************************************************************************************/
void Init_FXOS_Accelerometer()
{
	#if defined(FSL_FEATURE_SOC_LPI2C_COUNT) && (FSL_FEATURE_SOC_LPI2C_COUNT)
		lpi2c_master_config_t i2cConfig;
	#else
		i2c_master_config_t i2cConfig;
	#endif
	uint32_t i2cSourceClock;
	uint16_t i = 0;
	gpio_pin_config_t led_config = {kGPIO_DigitalOutput, 0, kGPIO_NoIntmode};
	uint8_t regResult = 0;

	BOARD_I2C_ReleaseBus();
	BOARD_InitI2C1Pins();
	BOARD_InitDebugConsole();
	SystemCoreClockUpdate();

	/* Init output LED GPIO. */
	GPIO_PinInit(EXAMPLE_LED_GPIO, EXAMPLE_LED_GPIO_PIN, &led_config);

	#if defined(FSL_FEATURE_SOC_LPI2C_COUNT) && (FSL_FEATURE_SOC_LPI2C_COUNT)
	i2cSourceClock = CLOCK_GetOscFreq();
	LPI2C_MasterGetDefaultConfig(&i2cConfig);
	LPI2C_MasterInit(BOARD_ACCEL_I2C_BASEADDR, &i2cConfig, i2cSourceClock);
	LPI2C_MasterTransferCreateHandle(BOARD_ACCEL_I2C_BASEADDR, &g_MasterHandle, FXOS_master_callback, NULL);
	#else
	i2cSourceClock = CLOCK_GetFreq(ACCEL_I2C_CLK_SRC);
	I2C_MasterGetDefaultConfig(&i2cConfig);
	I2C_MasterInit(BOARD_ACCEL_I2C_BASEADDR, &i2cConfig, i2cSourceClock);
	I2C_MasterTransferCreateHandle(BOARD_ACCEL_I2C_BASEADDR, &g_MasterHandle, NULL, NULL);
	#endif
	g_fxosHandle.base = BOARD_ACCEL_I2C_BASEADDR;
	g_fxosHandle.i2cHandle = &g_MasterHandle;
	/* Find sensor on board */
	for (i = 0; i < 4; i++)
	{
		g_fxosHandle.xfer.slaveAddress = g_sensor_address[i];
		if (FXOS_ReadReg(&g_fxosHandle, WHO_AM_I_REG, &regResult, 1) == kStatus_Success)
		{
			if(regResult == kFXOS_WHO_AM_I_Device_ID)
			{
				break;
			}
		}
		else /* Not found any sensor on board */
		{
			if (i == 3)
			{
				PRINTF("\r\nFailed to initialize the sensor device!");
				while (1)
				{
				}
			}
		}
	}
	/* Init sensor */
	FXOS_Init(&g_fxosHandle);
	/* Get sensor range */
	FXOS_ReadReg(&g_fxosHandle, XYZ_DATA_CFG_REG, &g_sensorRange, 1);
	if(g_sensorRange == 0x00)
	{
		g_dataScale = 2U;
	}
	else if(g_sensorRange == 0x01)
	{
		g_dataScale = 4U;
	}
	else if(g_sensorRange == 0x10)
	{
		g_dataScale = 8U;
	}
}

/* FUNCTION ************************************************************************************************************
 *
 * Function Name : read_sensor_data
 * Description   : Read the x,y,x axis data from accelerometer every 1 Ms
 *
 * END ****************************************************************************************************************/

void read_sensor_data()
{
#ifdef RUN_INFERENCE
	/* Take the chunks of no_of_samples as an input to model*/
	while(counter!=no_of_samples)
	{
		/* Check whether GPT interrupt occur*/
		if (true == gptIsrFlag)
		{
			counter++;
			gptIsrFlag = false;
			GPIO_PortToggle(EXAMPLE_LED_GPIO, 1u << 9);
			/* Get the accelerometer data for X,Y and Z axis */
			ACCEL_GetAccelData(&Ax, &Ay, &Az);
			/* Fill the data of X,Y,& Z axis to array of the no_of_samples which will be processed further */
			raw_data_x[(counter-1)] = Ax;
			raw_data_y[(counter-1)] = Ay;
			raw_data_z[(counter-1)] = Az;
		}
	}
#else
	/* Check whether GPT interrupt occur*/
	if (true == gptIsrFlag)
	{
		gptIsrFlag = false;
		GPIO_PortToggle(EXAMPLE_LED_GPIO, 1u << 9);
		/* Get the accelerometer data for X,Y and Z axis */
		ACCEL_GetAccelData(&Ax, &Ay, &Az);
		PRINTF("\r\nX=%d,Y=%d,Z=%d\n", Ax, Ay, Az);
	}
#endif
}
void Init_GPT()
{
	uint32_t gptFreq;
	gpt_config_t gptConfig;

	GPT_GetDefaultConfig(&gptConfig);
	/* Initialize GPT module */
	GPT_Init(EXAMPLE_GPT, &gptConfig);
	/* Divide GPT clock source frequency by 3 inside GPT module */
	GPT_SetClockDivider(EXAMPLE_GPT, 3);
	/* Get GPT clock frequency */
	gptFreq = EXAMPLE_GPT_CLK_FREQ;
	/* GPT frequency is divided by 3 inside module */
	gptFreq /= 3;
	/* Set both GPT modules to 1ms second duration */
	GPT_SetOutputCompareValue(EXAMPLE_GPT, kGPT_OutputCompare_Channel1, (gptFreq/1000));
	/* Enable GPT Output Compare1 interrupt */
	GPT_EnableInterrupts(EXAMPLE_GPT, kGPT_OutputCompare1InterruptEnable);
	/* Enable at the Interrupt */
	EnableIRQ(GPT_IRQ_ID);
	/* Start the timer to read over I2C every 1 ms */
	GPT_StartTimer(EXAMPLE_GPT);
}
