/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "app_inc.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

#define APP_TERMINAL_UART_RX_BUF_LEN   32u
static uint8_t        gAppTerUartRxBuf[APP_TERMINAL_UART_RX_BUF_LEN];
static RBUF_Handler_T gAppTerUartRxFifoHandle;
volatile bool         bAppRitTimeout = false;

/*******************************************************************************
 * Code
 ******************************************************************************/

void Terminal_Init(uint32_t baudrate)
{
    usart_config_t usartConfigStruct;
    rit_config_t ritConfigStruct;

    /* prepare the uart rx buffer and software flags. */
    RBUF_Init(&gAppTerUartRxFifoHandle, gAppTerUartRxBuf, APP_TERMINAL_UART_RX_BUF_LEN);
    bAppRitTimeout = false;

    /* setup uart. */
    USART_GetDefaultConfig(&usartConfigStruct);
    usartConfigStruct.baudRate_Bps = baudrate;
    usartConfigStruct.enableRx = true;
    usartConfigStruct.enableTx = true;
    usartConfigStruct.txWatermark = kUSART_TxFifo0;
    usartConfigStruct.rxWatermark = kUSART_RxFifo1;
    USART_Init(USART0, &usartConfigStruct, CLOCK_GetFreq(kCLOCK_Flexcomm0));

    /* enable uart rx interrupt. */
    USART_EnableInterrupts(USART0, kUSART_RxLevelInterruptEnable);
    NVIC_EnableIRQ(FLEXCOMM0_IRQn);

    /* setup rit timer. */
    ritConfigStruct.enableRunInDebug = true;
    RIT_Init(RIT, &ritConfigStruct);
    RIT_ClearCounter(RIT, true); /* Enable auto-clear when the counter reach to compare value.*/
    RIT_SetTimerCompare(RIT, CLOCK_GetFreq(kCLOCK_CoreSysClk)); /* Interval 1s. */
    NVIC_EnableIRQ(RIT_IRQn);

}

/* ISR entry for USART0. */
void FLEXCOMM0_IRQHandler(void)
{
    uint32_t flags = USART_GetStatusFlags(USART0);
    uint8_t  rxDat;

    /* rx available interrupt. */
    if (kUSART_RxFifoNotEmptyFlag == (kUSART_RxFifoNotEmptyFlag & flags) )
    {
        rxDat = USART_ReadByte(USART0);
        if ( !RBUF_IsFull(&gAppTerUartRxFifoHandle) )
        {
            RBUF_PutDataIn(&gAppTerUartRxFifoHandle, rxDat);
        }
    }

    USART_ClearStatusFlags(USART0, flags);
}

void Terminal_PutChar(uint8_t ch)
{
    USART_WriteBlocking(USART0, &ch, 1U);
}

void Terminal_PutString(char *str)
{
    while ('\0' != (*str))
    {
        Terminal_PutChar(*str);
        str++;
    }
}

/* ISR entry for RIT timer. */
void RIT_IRQHandler(void)
{
    uint32_t flags;

    flags = RIT_GetStatusFlags(RIT);

    bAppRitTimeout = true;

    RIT_ClearStatusFlags(RIT, flags);
    RIT_StopTimer(RIT); /* for one time trigger. */
}

bool Terminal_GetCharTimeout(uint8_t *rxDat, uint32_t ms)
{
    bool bRet = false;

    /* setup the alert timer. */
    bAppRitTimeout = false;
    if (ms > 0u)
    {
        RIT_StopTimer(RIT);
			  RIT_ClearStatusFlags(RIT, kRIT_TimerFlag);
        //RIT_ClearCounter(RIT, true); /* Enable auto-clear when the counter reach to compare value.*/
			  RIT->COUNTER = 0u; RIT->COMPVAL_H = 0u; /* reset the counter. */
        RIT_SetTimerCompare(RIT, CLOCK_GetFreq(kCLOCK_CoreSysClk) / 1000 * ms ); /* Interval 1s. */
        RIT_StartTimer(RIT);
    }

    while (1)
    {
        if ( !RBUF_IsEmpty(&gAppTerUartRxFifoHandle) )
        {
            *rxDat = RBUF_GetDataOut(&gAppTerUartRxFifoHandle);
            bRet = true;
            break; /* return with available rx data. */
        }
        if (bAppRitTimeout)
        {
            bRet = false;
            break; /* return without available rx data. */
        }
    }

    //RIT_ClearStatusFlags(RIT, kRIT_TimerFlag);
    RIT_StopTimer(RIT); /* for one time trigger. */
		//RIT_ClearCounter(RIT, true);
		RIT->COUNTER = 0u; RIT->COMPVAL_H = 0u; /* reset the counter. */
    return bRet;
}

/* EOF. */

