/*! *********************************************************************************
* \addtogroup Location and Navigation Service
* @{
 ********************************************************************************** */
/*!
* Copyright (c) 2016, Freescale Semiconductor, Inc.
* All rights reserved.
*
* \file location_and_navigation_source.c
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "ble_general.h"
#include "gatt_db_app_interface.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "MemManager.h"

#include "location_and_navigation_interface.h"

/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/***********************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
********************************************************************************** */

/***********************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
********************************************************************************** */
/*! Location and Navigation Service - Subscribed Client */
deviceId_t      mLns_ClientDeviceId = gInvalidDeviceId_c;
bool_t          mNotificationIncomplete = 0;

/***********************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
********************************************************************************** */
static void Lns_SendNotification (uint16_t handle);
static bleResult_t Lns_SetLnFeatures (uint16_t handle, lnsFeatures_t features);
static bleResult_t Lns_UpdateLocAndSpeedCharacteristic (uint16_t handle, lnsLocAndSpeed_t *pLocAndSpeed);
static bleResult_t Lns_UpdateNavigationCharacteristic (uint16_t handle, lnsNavigation_t *pNavigation);
static bleResult_t Lns_UpdatePositionQualityCharacteristic (uint16_t handle, lnsPositionQuality_t *pPositionQuality);

/***********************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
********************************************************************************** */
bleResult_t Lns_Start (lnsConfig_t *pServiceConfig)
{
    return Lns_SetLnFeatures(pServiceConfig->serviceHandle, pServiceConfig->lnsFeatures);
}

bleResult_t Lns_Stop (lnsConfig_t *pServiceConfig)
{    
    return Lns_Unsubscribe(pServiceConfig);
}

bleResult_t Lns_Subscribe(deviceId_t clientdeviceId)
{
    mLns_ClientDeviceId = clientdeviceId;
    return gBleSuccess_c;
}

bleResult_t Lns_Unsubscribe(lnsConfig_t *pServiceConfig)
{
    mLns_ClientDeviceId = gInvalidDeviceId_c;

    return gBleSuccess_c;
}

bleResult_t Lns_RecordLocationAndSpeedCharacteristic(uint16_t serviceHandle, lnsLocAndSpeed_t *pLocAndSpeed)
{
    uint16_t    handle;
    bleResult_t result;
    uint16_t    uuid = gBleSig_LocationAndSpeed_d;
    
    /* Get handle of characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType16_c, (bleUuid_t*)&uuid, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value and send indication */
    if (!Lns_UpdateLocAndSpeedCharacteristic(handle, pLocAndSpeed))
    {
        Lns_SendNotification(handle);
    }
    return gBleSuccess_c;
}

bleResult_t Lns_RecordNavigationCharacteristic(uint16_t serviceHandle, lnsNavigation_t *pNavigation)
{
    uint16_t    handle;
    bleResult_t result;
    uint16_t    uuid = gBleSig_Navigation_d;
    
    /* Get handle of characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType16_c, (bleUuid_t*)&uuid, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value and send indication */
    if (!Lns_UpdateNavigationCharacteristic(handle, pNavigation))
    {
        Lns_SendNotification(handle);
    }
    return gBleSuccess_c;
}

bleResult_t Lns_RecordPositionQualityCharacteristic(uint16_t serviceHandle, lnsPositionQuality_t *pPositionQuality)
{
    uint16_t    handle;
    bleResult_t result;
    uint16_t    uuid = gBleSig_PositionQuality_d;
    
    /* Get handle of characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType16_c, (bleUuid_t*)&uuid, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    Lns_UpdatePositionQualityCharacteristic(handle, pPositionQuality);

    return gBleSuccess_c;
}

/***********************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
********************************************************************************** */
static bleResult_t Lns_SetLnFeatures (uint16_t handle, lnsFeatures_t features)
{
    uint16_t  hValueLnFeature;
    uint16_t uuid = gBleSig_LnFeature_d;
    bleResult_t result;
  
      /* Get handle of characteristic */
    result = GattDb_FindCharValueHandleInService(handle,
                gBleUuidType16_c, (bleUuid_t*)&uuid, &hValueLnFeature);

    if (result != gBleSuccess_c)
        return result;
    
    return GattDb_WriteAttribute(hValueLnFeature, sizeof(lnsFeatures_t), (uint8_t*)&features);
}

static bleResult_t Lns_UpdateLocAndSpeedCharacteristic (uint16_t handle, lnsLocAndSpeed_t *pLocAndSpeed)
{
    uint8_t             charValue[20];
    uint8_t             index = 0x00;
    bleResult_t         result;
    ctsDayDateTime_t    utcTime;
    uint16_t            flags = 0x0000;

    if(!mNotificationIncomplete)
    {
        flags = pLocAndSpeed->lnsLocAndSpeedFlags & 0xFF9F;

        FLib_MemCpy(&charValue[index], &flags, sizeof(uint16_t));
        index += sizeof(uint16_t);
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_InstantaneousSpeedPresent_c)
        {
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsInstantaneousSpeed, sizeof(uint16_t));
            index += sizeof(uint16_t);
        }
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_TotalDistancePresent_c)
        {
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsTotalDistance, 3);
            index += 3;
        }
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_LocationPresent_c)
        {
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsLatitude, sizeof(int32_t));
            index += sizeof(int32_t);
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsLongitude, sizeof(int32_t));
            index += sizeof(int32_t);
        }
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_ElevationPresent_c)
        {
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsElevation, 3);
            index += 3;
        }
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_HeadingPresent_c)
        {
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsHeading, sizeof(uint16_t));
            index += sizeof(uint16_t);
        }
        mNotificationIncomplete = 1;
    }
    else
    {
        flags = pLocAndSpeed->lnsLocAndSpeedFlags & 0x0060;

        FLib_MemCpy(&charValue[index], &flags, sizeof(uint16_t));
        index += sizeof(uint16_t);
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_RollingTimePresent_c)
        {
            FLib_MemCpy(&charValue[index], &pLocAndSpeed->lnsRollingTime, sizeof(uint8_t));
            index += sizeof(uint8_t);
        }
        
        if(pLocAndSpeed->lnsLocAndSpeedFlags & gLns_UtcTimePresent_c)
        {
            utcTime = Cts_EpochToDayDateTime(pLocAndSpeed->lnsUtcTime);
            FLib_MemCpy(&charValue[index], &utcTime.dateTime, sizeof(ctsDateTime_t));
            index += sizeof(ctsDateTime_t);
        }
        mNotificationIncomplete = 0;
    }
    
    result = GattDb_WriteAttribute(handle, index, &charValue[0]);
    return result;
}

static bleResult_t Lns_UpdateNavigationCharacteristic (uint16_t handle, lnsNavigation_t *pNavigation)
{
    uint8_t             charValue[19];
    uint8_t             index = 0x00;
    bleResult_t         result;
    ctsDayDateTime_t    eta;

    FLib_MemCpy(&charValue[index], &pNavigation->lnsNavigationFlags, sizeof(uint16_t));
    index += sizeof(uint16_t);

    FLib_MemCpy(&charValue[index], &pNavigation->lnsBearing, sizeof(uint16_t));
    index += sizeof(uint16_t);
    
    FLib_MemCpy(&charValue[index], &pNavigation->lnsHeading, sizeof(uint16_t));
    index += sizeof(uint16_t);

    if(pNavigation->lnsNavigationFlags & gLns_RemainingDistancePresent_c)
    {
        FLib_MemCpy(&charValue[index], &pNavigation->lnsRemainingDistance, 3);
        index += 3;
    }

    if(pNavigation->lnsNavigationFlags & gLns_RemainingVerticalDistancePresent_c)
    {
        FLib_MemCpy(&charValue[index], &pNavigation->lnsRemainingVerticalDistance, 3);
        index += 3;
    }
    
    if(pNavigation->lnsNavigationFlags & gLns_EstimatedTimeOfArrivalPresent_c)
    {
        eta = Cts_EpochToDayDateTime(pNavigation->lnsETA);
        FLib_MemCpy(&charValue[index], &eta.dateTime, sizeof(ctsDateTime_t));
        index += sizeof(ctsDateTime_t);
    }
    
    result = GattDb_WriteAttribute(handle, index, &charValue[0]);
    return result;
}

static bleResult_t Lns_UpdatePositionQualityCharacteristic (uint16_t handle, lnsPositionQuality_t *pPositionQuality)
{
    uint8_t             charValue[16];
    uint8_t             index = 0x00;
    bleResult_t         result;
    
    FLib_MemCpy(&charValue[index], &pPositionQuality->lnsPositionQualityFlags, sizeof(uint16_t));
    index += sizeof(uint16_t);
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_NoOfBeaconsInSolutionPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsNumberOfBeaconsInSolution, sizeof(uint8_t));
        index += sizeof(uint8_t);
    }
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_NoOfBeaconsInViewPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsNumberOfBeaconsInView, sizeof(uint8_t));
        index += sizeof(uint8_t);
    }
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_TimeToFirstFixPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsTimeToFirstFix, sizeof(uint16_t));
        index += sizeof(uint16_t);
    }
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_EHPEPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsEHPE, sizeof(uint32_t));
        index += sizeof(uint32_t);
    }
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_EVPEPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsEVPE, sizeof(uint32_t));
        index += sizeof(uint32_t);
    }
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_HDOPPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsHDOP, sizeof(uint8_t));
        index += sizeof(uint8_t);
    }
    
    if(pPositionQuality->lnsPositionQualityFlags & gLns_VDOPPresent_c)
    {
        FLib_MemCpy(&charValue[index], &pPositionQuality->lnsVDOP, sizeof(uint8_t));
        index += sizeof(uint8_t);
    }
    
    result = GattDb_WriteAttribute(handle, index, &charValue[0]);
    return result;
}

static void Lns_SendProcedureResponse (lnsConfig_t *pServiceConfig, gattServerAttributeWrittenEvent_t* pEvent)
{
    uint8_t             rspSize = 3;
    lnsProcedure_t*     pResponse;
    lnsProcedure_t*     pProcedure = (lnsProcedure_t*)pEvent->aValue;
    uint8_t             procDataLength = pEvent->cValueLength - sizeof(lnsOpCode_t);
    uint16_t            handleCccd;
    uint16_t            handle;
    uint16_t            uuid = gBleSig_Navigation_d;

    /* Allocate buffer for biggest response */
    pResponse = MEM_BufferAlloc(rspSize + 4);
    
    if (pResponse == NULL)
    {
        return;
    }

    pResponse->opCode = gLns_ResponseCode_c;
    pResponse->procedureData.response.reqOpCode = pProcedure->opCode;
    
    switch (pProcedure->opCode)
    {
        case gLns_SetCumulativeValue_c:
        {
            if (procDataLength == 3) /* size of INT24 */
            {
                pServiceConfig->pUserData->lnsLocAndSpeed.lnsTotalDistance = pProcedure->procedureData.lnsTotalDistance;
                pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            }
            else
            {
                pResponse->procedureData.response.rspValue = gErrCodeLnsInvalidParameter_c;
            }
            break;
        }
        
        case gLns_MaskLocAndSpeedContent_c:
        {
            if (procDataLength == (sizeof(uint16_t)))
            {
                pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            }
            else
            {
                pResponse->procedureData.response.rspValue = gErrCodeLnsInvalidParameter_c;
            }
            break;
        }
        
        case gLns_NavigationControl_c:
        {
            if (procDataLength == (sizeof(uint8_t)))
            {
                /* Get handle of characteristic */
                if(GattDb_FindCharValueHandleInService(pServiceConfig->serviceHandle,
                    gBleUuidType16_c, (bleUuid_t*)&uuid, &handle) != gBleSuccess_c)
                    return;
                
                /* Get handle of CCCD */
                if (GattDb_FindCccdHandleForCharValueHandle(handle, &handleCccd) != gBleSuccess_c)
                    return;
              
                if(pProcedure->procedureData.lnsNavigationControl == gLnsStopNavigation_c)
                {
                    Gap_SaveCccd(mLns_ClientDeviceId, handleCccd, gCccdEmpty_c);
                }
                else if (pProcedure->procedureData.lnsNavigationControl == gLnsStartNavigationFirstWaypoint_c)
                {
                    Gap_SaveCccd(mLns_ClientDeviceId, handleCccd, (gCccdNotification_c));
                }
                else if (pProcedure->procedureData.lnsNavigationControl == gLnsPauseNavigation_c)
                {
                    Gap_SaveCccd(mLns_ClientDeviceId, handleCccd, (gCccdEmpty_c));
                }
                else if (pProcedure->procedureData.lnsNavigationControl == gLnsContinueNavigation_c)
                {
                    Gap_SaveCccd(mLns_ClientDeviceId, handleCccd, (gCccdNotification_c));
                }
                else if (pProcedure->procedureData.lnsNavigationControl == gLnsSkipWaypoint_c)
                {

                }
                else if (pProcedure->procedureData.lnsNavigationControl == gLnsNearestWaypoint_c)
                {

                }
                pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            }
            else
            {
                pResponse->procedureData.response.rspValue = gErrCodeLnsInvalidParameter_c;
            }
            break;
        }
        
        case gLns_RequestNumberOfRoutes_c:
        {
            pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            pResponse->procedureData.response.rspData.lnsNumberOfRoutes = 0x0002;
            rspSize += sizeof(uint16_t);
            break;
        }
        
        case gLns_RequestNameOfRoutes_c:
        {
            pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            char* name = "asdf";
            FLib_MemCpy(pResponse->procedureData.response.rspData.lnsNameOfRoute, name, 4);
            rspSize += sizeof(uint32_t);
            break;
        }
        
        case gLns_SelectRoute_c:
        {
            pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            pResponse->procedureData.response.rspData.lnsSelectedRoute = 0x0001;
            break;
        }
          
        case gLns_SetElevation_c:
        {
            if (procDataLength == 3) /* size of INT24 */
            {
                pServiceConfig->pUserData->lnsLocAndSpeed.lnsElevation = pProcedure->procedureData.lnsElevation;
                pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            }
            else
            {
                pResponse->procedureData.response.rspValue = gErrCodeLnsInvalidParameter_c;
            }
            break;
        }
        
        case gLns_SetFixRate_c:
        {
            if (procDataLength == sizeof(uint8_t))
            {
                pResponse->procedureData.response.rspValue = gErrCodeNoError_c;
            }
            else
            {
                pResponse->procedureData.response.rspValue = gErrCodeLnsInvalidParameter_c;
            }
            break;
        }
        
        default:
        {
            pResponse->procedureData.response.rspValue = gErrCodeLnsOpCodeNotSupported_c;
            break;
        }
        
    }

    /* Write response in characteristic */
    GattDb_WriteAttribute(pEvent->handle, rspSize, (uint8_t*) pResponse);

    /* Free buffer */
    MEM_BufferFree(pResponse);

    /* Indicate value to client */
    GattServer_SendIndication(mLns_ClientDeviceId, pEvent->handle);
}

void Lns_ControlPointHandler (lnsConfig_t *pServiceConfig, gattServerAttributeWrittenEvent_t *pEvent)
{
    bool_t  fIndicationActive = FALSE;
    uint16_t  handleCccd;
   
    /* Get handle of CCCD */
    if (GattDb_FindCccdHandleForCharValueHandle(pEvent->handle, &handleCccd) != gBleSuccess_c)
        return;

    /* Check if indications are properly configured */
    Gap_CheckIndicationStatus(mLns_ClientDeviceId, handleCccd, &fIndicationActive);

    if(!fIndicationActive)
    {
        GattServer_SendAttributeWrittenStatus(mLns_ClientDeviceId, pEvent->handle,
                                                 gAttErrCodeCccdImproperlyConfigured_c);
        return;
    }

    /* Procedure received successfully */
    GattServer_SendAttributeWrittenStatus(mLns_ClientDeviceId, pEvent->handle,
                                                     gAttErrCodeNoError_c);

    Lns_SendProcedureResponse(pServiceConfig, pEvent);
}

static void Lns_SendNotification (uint16_t handle)
{
    uint16_t    hCccd;
    bool_t      isNotificationActive;

    /* Get handle of CCCD */
    if (GattDb_FindCccdHandleForCharValueHandle(handle, &hCccd) != gBleSuccess_c)
        return;

    if (gBleSuccess_c == Gap_CheckNotificationStatus
        (mLns_ClientDeviceId, hCccd, &isNotificationActive) &&
        TRUE == isNotificationActive)
    {
        GattServer_SendNotification(mLns_ClientDeviceId, handle);
    }
}

/*! *********************************************************************************
* @}
********************************************************************************** */
