/*! *********************************************************************************
* \defgroup Location and Navigation Service
* @{
*********************************************************************************** */
/*!
* Copyright (c) 2016, Freescale Semiconductor, Inc.
* All rights reserved.
*
* \file location_and_navigation_interface.h
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef _LOCATION_AND_NAGIATION_INTERFACE_H_
#define _LOCATION_AND_NAGIATION_INTERFACE_H_

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "current_time_interface.h"

/************************************************************************************
*************************************************************************************
* Public constants & macros
*************************************************************************************
************************************************************************************/
#define Lns_AllFeatures (gLns_InstantaneousSpeedSupported_c | gLns_TotalDistanceSupported_c |\
                        gLns_LocationSupported_c | gLns_ElevationSupported_c | gLns_HeadingSupported_c |\
                        gLns_RollingTimeSupported_c | gLns_UtcTimeSupported_c | gLns_RemainingDistanceSupported_c |\
                        gLns_RemainingVerticalDistanceSupported_c | gLns_EstmatedTimeOfArrivalSupported_c |\
                        gLns_NumberOfBeaconsInSolutionSupported_c | gLns_NumberOfBeaconsInViewSupported_c |\
                        gLns_TimeToFirstFixSupported_c | gLns_EstimatedHorizontalPositionErrorSupported_c |\
                        gLns_EstimatedVerticalPositionErrorSupported_c | gLns_HorizontalDilutionOfPrecisionSupported_c |\
                        gLns_VerticalDilutionOfPrecisionSupported_c | gLns_LocationAndSpeedCharContentMaskingSupported_c |\
                        gLns_FixRateSettingSupported_c | gLns_ElevationSettingSupported_c | gLns_PositionStatusSupported_c )

#define Lns_MandatoryFeatures    0x00
/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/

typedef enum
{
    gLns_NoFeature_c                                    = 0x00,
    gLns_InstantaneousSpeedSupported_c                  = BIT0,
    gLns_TotalDistanceSupported_c                       = BIT1,
    gLns_LocationSupported_c                            = BIT2,
    gLns_ElevationSupported_c                           = BIT3,
    gLns_HeadingSupported_c                             = BIT4,
    gLns_RollingTimeSupported_c                         = BIT5,
    gLns_UtcTimeSupported_c                             = BIT6,
    gLns_RemainingDistanceSupported_c                   = BIT7,
    gLns_RemainingVerticalDistanceSupported_c           = BIT8,
    gLns_EstmatedTimeOfArrivalSupported_c               = BIT9,
    gLns_NumberOfBeaconsInSolutionSupported_c           = BIT10,
    gLns_NumberOfBeaconsInViewSupported_c               = BIT11,
    gLns_TimeToFirstFixSupported_c                      = BIT12,
    gLns_EstimatedHorizontalPositionErrorSupported_c    = BIT13,
    gLns_EstimatedVerticalPositionErrorSupported_c      = BIT14,
    gLns_HorizontalDilutionOfPrecisionSupported_c       = BIT15,
    gLns_VerticalDilutionOfPrecisionSupported_c         = BIT16,
    gLns_LocationAndSpeedCharContentMaskingSupported_c  = BIT17,
    gLns_FixRateSettingSupported_c                      = BIT18,
    gLns_ElevationSettingSupported_c                    = BIT19,
    gLns_PositionStatusSupported_c                      = BIT20,
} lnsFeatures_tag;

typedef uint32_t lnsFeatures_t;

typedef enum
{
    gLns_NoFlags                                        = 0x00,
    gLns_InstantaneousSpeedPresent_c                    = BIT0,
    gLns_TotalDistancePresent_c                         = BIT1,
    gLns_LocationPresent_c                              = BIT2,
    gLns_ElevationPresent_c                             = BIT3,
    gLns_HeadingPresent_c                               = BIT4,
    gLns_RollingTimePresent_c                           = BIT5,
    gLns_UtcTimePresent_c                               = BIT6,
} lnsFlags_tag;

typedef enum
{
    gLns_NoPosition_c                                   = 0x00,
    gLns_PositionOk_c                                   = BIT7,
    gLns_EstimatedPosition_c                            = BIT8,
    gLns_LastKnownPosition_c                            = BIT7 | BIT8
} lnsPositionStatus_tag;

typedef enum
{
    gLns_SpeedAndDistance2D_c                           = 0x00,
    gLns_SpeedAndDistance3D_c                           = BIT9
} lnsSpeedAndDistanceFormat_tag;

typedef enum
{
    gLns_PositioningSystem_c                            = 0x00,
    gLns_BarometricAirPressure_c                        = BIT10,
    gLns_DatabaseService_c                              = BIT11,
    gLns_OtherElevationSource_c                         = BIT10 | BIT11
} lnsElevationSource_tag;

typedef enum
{
    gLns_HeadingBasedOnMovement_c                       = 0x00,
    gLns_HeadingBasedOnMagneticCompass_c                = BIT12
} lnsHeadingSource_tag;

typedef uint16_t lnsLocAndSpeedFlags_t;

typedef PACKED_STRUCT lnsLocAndSpeed_tag
{
    lnsLocAndSpeedFlags_t       lnsLocAndSpeedFlags;
    uint16_t                    lnsInstantaneousSpeed;
    uint32_t                    lnsTotalDistance;
    int32_t                     lnsLatitude;
    int32_t                     lnsLongitude;
    int32_t                     lnsElevation;
    uint16_t                    lnsHeading;
    uint8_t                     lnsRollingTime;
    uint32_t                    lnsUtcTime;
} lnsLocAndSpeed_t;

typedef enum
{
    gLns_NoOfBeaconsInSolutionPresent_c = BIT0,
    gLns_NoOfBeaconsInViewPresent_c     = BIT1,
    gLns_TimeToFirstFixPresent_c        = BIT2,
    gLns_EHPEPresent_c                  = BIT3,
    gLns_EVPEPresent_c                  = BIT4,
    gLns_HDOPPresent_c                  = BIT5,
    gLns_VDOPPresent_c                  = BIT6,
} lnsPositionQualityFlags_tag;

typedef uint16_t lnsPositionQualityFlags_t;

typedef struct lnsPositionQuality_tag
{
    lnsPositionQualityFlags_t   lnsPositionQualityFlags;
    uint8_t                     lnsNumberOfBeaconsInSolution;
    uint8_t                     lnsNumberOfBeaconsInView;
    uint16_t                    lnsTimeToFirstFix;
    uint32_t                    lnsEHPE;
    uint32_t                    lnsEVPE;
    uint8_t                     lnsHDOP;
    uint8_t                     lnsVDOP;
} lnsPositionQuality_t;

typedef enum
{
    gLns_RemainingDistancePresent_c              = BIT0,
    gLns_RemainingVerticalDistancePresent_c      = BIT1,
    gLns_EstimatedTimeOfArrivalPresent_c         = BIT2,
    gLns_WaypointReached_c                       = BIT7,
    gLns_DestinationReached_c                    = BIT8
} lnsNavigationFlags_tag;

typedef enum
{
    gNavigation_NoPosition_c            = 0x00,
    gNavigation_PositionOk_c            = BIT3,
    gNavigation_EstimatedPosition_c     = BIT4,
    gNavigation_LastKnownPosition_c     = BIT3 | BIT4
} lnsNavigationPositionStatus_tag;

typedef enum
{
    gNavigation_HeadingBasedOnMovement_c        = 0x00,
    gNavigation_HeadingBasedOnMagneticCompass_c = BIT5
} lnsNavigationHeadingSource_tag;

typedef enum
{
    gLns_ToWaypoint_c           = 0x00,
    gLns_ToDestination_c        = BIT6
} lnsNavigationIndicatorType_tag;

typedef uint16_t lnsNavigationFlags_t;

typedef struct lnsNavigation_tag
{
    lnsNavigationFlags_t        lnsNavigationFlags;
    uint16_t                    lnsBearing;
    uint16_t                    lnsHeading;
    uint32_t                    lnsRemainingDistance;
    int32_t                     lnsRemainingVerticalDistance;
    uint32_t                    lnsETA;
} lnsNavigation_t;

typedef uint16_t lnsLocAndSpeedMask_t;

typedef struct lnsUserData_tag
{
    lnsLocAndSpeed_t            lnsLocAndSpeed;
    lnsPositionQuality_t        lnsPositionQuality;
    lnsNavigation_t             lnsNavigation;
    lnsLocAndSpeedMask_t        lnsMask;
} lnsUserData_t;

/*! Location and Navigation Service - Procedures Op Codes */
typedef enum
{
    gLns_SetCumulativeValue_c =           0x01,
    gLns_MaskLocAndSpeedContent_c =       0x02,
    gLns_NavigationControl_c =            0x03,
    gLns_RequestNumberOfRoutes_c =        0x04,
    gLns_RequestNameOfRoutes_c =          0x05,
    gLns_SelectRoute_c =                  0x06,
    gLns_SetFixRate_c =                   0x07,
    gLns_SetElevation_c =                 0x08,
    gLns_ResponseCode_c =                 0x20,
} lnsOpCode_tag;

typedef uint8_t lnsOpCode_t;

typedef enum
{
    gLnsInstantaneousSpeedOff_c =       BIT0,
    gLnsTotalDistanceOff_c =            BIT1,
    gLnsLocationOff_c =                 BIT2,
    gLnsElevationOff_c =                BIT3,
    gLnsHeadingOff_c =                  BIT4,
    gLnsRollingTimeOff_c =              BIT5,
    gLnsUtcTimeOff_c =                  BIT6,
} lnsLocAndSpeedMask_tag;

typedef enum
{
    gLnsStopNavigation_c = 0x00,
    gLnsStartNavigationFirstWaypoint_c,
    gLnsPauseNavigation_c,
    gLnsContinueNavigation_c,
    gLnsSkipWaypoint_c,
    gLnsNearestWaypoint_c,
} lnsNavigationControl_tag;

/*! Location and Navigation Service - ATT Error Codes */
typedef enum
{
    gErrCodeNoError_c =                 0x01,
    gErrCodeLnsOpCodeNotSupported_c =   0x02,
    gErrCodeLnsInvalidParameter_c =     0x03,
    gErrCodeLnsOperationFailed_c =      0x04,
}cpsRspValue_tag;

typedef uint8_t lnsRspValue_t;

typedef struct lnsResponse_tag{
    lnsOpCode_t    reqOpCode;
    lnsRspValue_t  rspValue;
    union {
        uint16_t    lnsNumberOfRoutes;
        uint8_t     lnsNameOfRoute[4];
        uint16_t    lnsSelectedRoute;
    } rspData;
}lnsResponse_t;

/*! Location and Navigation Service - SC Procedure */
typedef PACKED_STRUCT lnsProcedure_tag {
    lnsOpCode_t    opCode;
    union 
    {
        uint32_t        lnsTotalDistance;
        uint16_t        lnsMaskLocAndSpeedCharContent;
        uint8_t         lnsNavigationControl;
        uint16_t        lnsRouteNumber;
//        uint16_t        lns
        uint16_t        lnsSelectRoute;
        uint8_t         lnsFixRate;
        int32_t         lnsElevation;
        lnsResponse_t   response;
    } procedureData;
} lnsProcedure_t;

/*! Location and Navigation Service - Configuration */
typedef struct lnsConfig_tag
{
    uint16_t                    serviceHandle;
    lnsFeatures_t               lnsFeatures;
    lnsUserData_t               *pUserData;
} lnsConfig_t;

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/*!**********************************************************************************
* \brief        Starts Location and Navigation Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lns_Start (lnsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Stops Location and Navigation Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lns_Stop (lnsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Subscribes a GATT client to the Location and Navigation service
*
* \param[in]    clientdeviceId  Client Id in Device DB.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lns_Subscribe(deviceId_t clientdeviceId);

/*!**********************************************************************************
* \brief        Unsubscribes a GATT client from the Location and Navigation service
*
* \param[in]    pServiceConfig  Pointer to structure that contains server
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lns_Unsubscribe(lnsConfig_t *pServiceConfig);

bleResult_t Lns_RecordLocationAndSpeedCharacteristic(uint16_t serviceHandle, lnsLocAndSpeed_t *pLocAndSpeed);
bleResult_t Lns_RecordNavigationCharacteristic(uint16_t serviceHandle, lnsNavigation_t *pNavigation);
bleResult_t Lns_RecordPositionQualityCharacteristic(uint16_t serviceHandle, lnsPositionQuality_t *pPositionQuality);
void Lns_ControlPointHandler (lnsConfig_t *pServiceConfig, gattServerAttributeWrittenEvent_t *pEvent);

#ifdef __cplusplus
}
#endif 

#endif /* _LOCATION_AND_NAGIATION_INTERFACE_H_ */

/*! *********************************************************************************
* @}
********************************************************************************** */