/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _QN908X_PROP_H_
#define _QN908X_PROP_H_

#ifdef __cplusplus
    extern "C" {
#endif

#include "fsl_common.h"
 
/*!
 * @addtogroup PROP API
 * @{
 */
        
/*******************************************************************************
 * Definitions
 *****************************************************************************/
#define PROP_RADIOFREQ_BASE_2400MHZ  (2400+1)
      
#define PROP_RADIOMODEBT0P5H0P5  128
#define PROP_RADIOMODEBT0P5H0P32 82
#define PROP_RADIOMODEBT0P5H0P7  180
      
      
/*! @brief PROP Radio operation */
typedef enum _prop_radio_operation
{
    kPROP_RadioFreqTx   = 0,    /*!< 1 MBit datarate Tx mode */
    kPROP_RadioFreqRxMode1 = 1,    /*!< 1 MBit,500Kbit or 250Kbit datarate Rx mode */
    kPROP_RadioFreqRxMode2 = 2,    /*!< 2 MBit datarate Rx mode */
} prop_radio_operation_t;

typedef enum _prop_radio_mode
{
    kPROP_RadioModeBt0p5h0p5  = 128,  /*!< BT=0.5, h=0.5 [BLE at 1MBPS data rate; CS4 at 250KBPS data rate] */
    kPROP_RadioModeBt0p5h0p32 = 82,   /*!< BT=0.5, h=0.32*/
    kPROP_RadioModeBt0p5h0p7  = 180,  /*!< BT=0.5, h=0.7 [CS1 at 500KBPS data rate] */
}prop_radio_mode_t;

/*! @brief PROP Data Rate selection */
typedef enum _prop_data_rate
{
    kPROP_DataRate250KBit   = 0,  /*!< PROP 250 KBit datarate */
    kPROP_DataRate500KBit   = 1,  /*!< PROP 500 KBit datarate */
    kPROP_DataRate1MBit     = 2,  /*!< PROP 1   MBit datarate */
    kPROP_DataRate2MBit     = 3,  /*!< PROP 2   MBit datarate */
} prop_data_rate_t;

/*! @brief PROP preamable size selection */
typedef enum _prop_preamble_size
{
    kPROP_PreamableSize1Byte   = 0,  /*!< PROP preamable size is 1 Byte */
    kPROP_PreamableSize2Byte   = 1,  /*!< PROP preamable size is 2 Byte */
    kPROP_PreamableSize3Byte   = 2,  /*!< PROP preamable size is 3 Byte */
    kPROP_PreamableSize4Byte   = 3,  /*!< PROP preamable size is 4 Byte */
    kPROP_PreamableSize5Byte   = 4,  /*!< PROP preamable size is 5 Byte */
    kPROP_PreamableSize6Byte   = 5,  /*!< PROP preamable size is 6 Byte */
    kPROP_PreamableSize7Byte   = 6,  /*!< PROP preamable size is 7 Byte */
    kPROP_PreamableSize8Byte   = 7,  /*!< PROP preamable size is 8 Byte */
} prop_preamble_size_t;

/*! @brief PROP Access Address size selection */
typedef enum _prop_access_address_size
{
    kPROP_AccessAddrSize2Byte   = 0, /*!< PROP Access Address size is 2 Byte */
    kPROP_AccessAddrSize3Byte   = 1, /*!< PROP Access Address size is 3 Byte */
    kPROP_AccessAddrSize4Byte   = 2, /*!< PROP Access Address size is 4 Byte */
    kPROP_AccessAddrSize5Byte   = 3, /*!< PROP Access Address size is 5 Byte */
} prop_access_address_size_t;

/*! @brief PROP crc size selection */
typedef enum _prop_crc_size
{
    kPROP_CrcSize0Bit  = 0,  /*!< PROP crc not used  */
    kPROP_CrcSize8Bit  = 1,  /*!< PROP crc size is 1 Byte */
    kPROP_CrcSize16Bit = 2,  /*!< PROP crc size is 2 Byte */
    kPROP_CrcSize24Bit = 3,  /*!< PROP crc size is 3 Byte */
} prop_crc_size_t;

/*! @brief PROP crc startbyte selection */
typedef enum _prop_crc_startbyte
{
    kPROP_CrcStartByteAccess  = 0,  /*!< PROP crc not used  */
    kPROP_CrcStartBytePdu     = 1,  /*!< PROP crc size is 1 Byte */
} prop_crc_startbyte_t;

/*! @brief PROP bit order selection */
typedef enum _prop_bit_order
{
    kPROP_BitOrderLsb   = 0, /*!< Bit order of the LS Bit first */
    kPROP_BitOrderMsb   = 1, /*!< Bit order of the MS Bit first */
} prop_bit_order_t;

/*! @brief PROP modem mode */
typedef enum _prop_modem_mode
{
    kPROP_UnModulation   = 0, /*!< modulation */
    kPROP_Modulation   = 1, /*!< Unmodulation*/
} prop_modem_mode_t;

/*! @brief PROP modem mode */
typedef enum _prop_pattern_type
{
    kPROP_PatternType0   = 0, /*!< Pattern of All 0 bits*/
    kPROP_PatternType1   = 1, /*!< Pattern of All 1 bits*/
    kPROP_PatternType2   = 2, /*!< Pattern of alternating bits 01010101*/
    kPROP_PatternType3   = 3, /*!< Pattern of alternating bits 10101010*/
    kPROP_PatternType4   = 4, /*!< Pattern of alternating bits 00001111*/
    kPROP_PatternType5   = 5, /*!< Pattern of alternating bits 11110000*/
    kPROP_PatternType6   = 6, /*!< Pseudo-Random bit sequence 9*/
} prop_pattern_type_t;


/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @brief   Initialize PROP module.
 * 
 * This API sets up prop module enviroment, initialize control parameter, eanble its'clock,
 *
 * @warning: It should be called firstly.
 *
 */
void PROP_Init(PROP_Type *base);

/*!
 * @brief   Deinitializes PROP module.
 *
 * This API disables transfer request and gates its clock.
 */
void PROP_DeInit(PROP_Type *base);

/*!
 * @brief  Start transmitting data.
 *
 * @param[in]  base PROP peripheral base address.
 * 
 * @description This API start Tx
 */
void PROP_StartTx(PROP_Type *base);

/*!
 * @brief  Start transmitting data.
 *
 * @param[in]  base PROP peripheral base address.
 *
 * @description This API start Rx
 */
void PROP_StartRx(PROP_Type *base);


/*!
 * @brief Stop Tx actions.
 * 
 * @param[in]  base PROP peripheral base address.
 *
 * @description This API gives up and exit from Tx execution.
 */
void PROP_StopTx(PROP_Type *base);


/*!
 * @brief Stop Rx actions.
 * 
 * @param[in]  base PROP peripheral base address.
 *
 * @description This API gives up and exit from Rx execution.
 */
void PROP_StopRx(PROP_Type *base);

/*!
 * @brief Set payload Length.
 * 
 * @param[in]  base PROP peripheral base address.
 * @param[in]  payloadLen the length of payload.
 *
 * @description This API set the payload length.
 */
void PROP_SetPayloadLen(PROP_Type *base, uint16_t payloadLen);

/*!
 * @brief Get payload Length.
 * 
 * @param[in]  base PROP peripheral base address.
 * @return the length of payload
 * @description This API get the payload length.
 */
uint16_t PROP_GetPayloadLen(PROP_Type *base);

/*!
 * @brief Get PROP TX status.
 * 
 * @param[in]  base PROP peripheral base address.
 * @return  True if busy.
 * @description This API get Tx status.
 */
bool PROP_GetTxStatus(PROP_Type *base);

/*!
 * @brief Get PROP crc status.
 * 
 * @param[in]  base PROP peripheral base address.
 * @return  True if CRC correct.
 * @description This API get the crc status.
 */
bool PROP_GetCrcStatus(PROP_Type *base);

/*!
 * @brief Get RSSI value 
 *
 * @param[in]  base PROP peripheral base address.
 * @return RSSI value.
 */
uint8_t PROP_GetRssi(PROP_Type *base);

/*!
 * @brief  Set Packet Config.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[in]  preambelSize : preamble size
 * @param[in]  accessAddress : access address size
 * @param[in]  bitOrder :  bitOrder
 *
 * @description This API sets Packet Config to PROP module.
 */
void PROP_SetPacketConfig(PROP_Type *base, uint8_t preambelSize, uint8_t accessAddress,uint8_t bitOrder);


/*!
 * @brief  Get Packet Config.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[out]  preambelSize Pointer to the location to store the preamble size.
 * @param[out]  accessAddress Pointer to the location to store the access address size
 * @param[out]  bitOrder :  Pointer to the location to store the bitOrder
 *
 * @description This API gets Packet Config to PROP module.
 */
void PROP_GetPacketConfig(PROP_Type *base, uint8_t *preambelSize, uint8_t *accessAddress, uint8_t *bitOrder);

/*!
 * @brief  Set network access address.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[in]  pAcessAddress    Pointer to the location to store access address; 
 *
 * @description This API sets net access address to PROP module.
 */
void PROP_SetNetworkAddress(PROP_Type *base, uint8_t *pAccessAddress);

/*!
 * @brief  Get network access address.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[out]  pAcessAddress    Pointer to the location to store access address; 
 *
 * @description This API gets net access address to PROP module.
 */
void PROP_GetNetworkAddress(PROP_Type *base, uint8_t *pAccessAddress);

/*!
 * @brief  Set channel number.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[in]  channelNum the channel number 0,1,2......79
 * @param[in] radioOp  the radio operation: TX, RX mode2, RX mode1
 * @description This API sets channel number to PROP module.
 */
void PROP_SetChannelNumber(PROP_Type *base, uint8_t channelNum, prop_radio_operation_t radioOp);

/*!
 * @brief  Get channel number.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[in] radioOp  the radio operation: TX, RX mode2, RX mode1
 * @return  the channel number 0,1,2......79
 *
 * @description This API Gets channel number to PROP module.
 */
uint8_t PROP_GetChannelNumber(PROP_Type *base, prop_radio_operation_t radioOp);

/*!
 * @brief  Set Crc config.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[in]  crcSize  the size of crc(1 byte,2 byte,3 byte)
 * @param[in]  crcStartByte  the start byte of crc(0-enable check access address 1-disable check access address)
 * @param[in]  crcSeed  the crc seed
 * @description This API sets CRC config to PROP module.
 */
void PROP_SetCrcConfig(PROP_Type *base, uint8_t crcSize, uint8_t crcStartByte, uint32_t crcSeed);


/*!
 * @brief  Get Crc config.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[out]  crcSize Pointer to the location to store the crc size.
 * @param[out]  crcStartByte Pointer to the location to store the crc start byte.
 * @param[out]  crcSeed  Pointer to the location to store the crc seed.
 * @description This API gets CRC config to PROP module.
 */
void PROP_GetCrcConfig(PROP_Type *base, uint8_t *crcSize, uint8_t *crcStartByte, uint32_t *crcSeed);


/*!
 * @brief  Set Radio config.
 *
 * @param[in]  base PROP peripheral base address.
 * @param[in]  radioMode the mode of radio
 * @param[in]  dataRate   the data rate
 * @description This API sets Radio config to PROP module.
 */
void PROP_RadioConfig(PROP_Type *base, prop_radio_mode_t radioMode, prop_data_rate_t dataRate);


/*!
 * @brief Modem Tx actions.
 *
 * @param[in] base PROP peripheral base address.
 * @param[in] mode the mode of modem.
 * @param[in] payload_type the payload type.
 */
void PROP_ModTx(PROP_Type *base,prop_modem_mode_t mode,prop_pattern_type_t payload_type);

/*!
 * @brief set the Modem to idle.
 *
 * @param[in] base PROP peripheral base address.
 */
void PROP_ModIdle(PROP_Type *base);

/*!
 * @brief Modem Rx actions.
 *
 * @param[in] base PROP peripheral base address.
 */
void PROP_ModRx(PROP_Type *base);


/*! @} PROP API */ 
#endif /* _QN908X_PROP_H_ */

